/*
 * File:     TimePanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator;

import mpi.eudico.client.annotator.viewer.*;

import mpi.eudico.client.mediacontrol.ControllerEvent;
import mpi.eudico.client.mediacontrol.StopEvent;
import mpi.eudico.client.mediacontrol.TimeEvent;

import mpi.eudico.client.util.TimeFormatter;

import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.ButtonGroup;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.SwingUtilities;


/**
 * DOCUMENT ME!
 * $Id: TimePanel.java,v 1.4 2006/02/27 15:06:47 hasloe Exp $
 * @author $Author: hasloe $
 * @version $Revision: 1.4 $
 */
public class TimePanel extends AbstractViewer implements ActionListener {
    private JLabel timeLabel;
    private String STRCROSSHAIRTIMEINPUTBOX1 = "";
    private String STRCROSSHAIRTIMEINPUTBOX2 = "";
    private String STRCROSSHAIRTIMEINPUTBOX3 = "";
    private String STRCROSSHAIRTIMEINPUTBOX4 = "";
    private String STRCROSSHAIRTIMEINPUTBOX5 = "";

    // popup for time format
    private JPopupMenu popup;
    private ButtonGroup formatBG;
    private JMenu formatMenu;
    private JRadioButtonMenuItem hhmmssmsMI;
    private JRadioButtonMenuItem timecodePalMI;
    private JRadioButtonMenuItem timecodeNtscMI;
    private final int TC = 0;
    private final int TC_PAL = 1;
    private final int TC_NTSC = 2;
    private int mode = TC;

    /**
     * Creates a new TimePanel instance
     */
    public TimePanel() {
        setLayout(new BorderLayout());

        timeLabel = new JLabel("00:00:00.000");
        timeLabel.setFont(Constants.SMALLFONT);

        addMouseListener(new TimeLabelMouseHandler());

        add(timeLabel, BorderLayout.CENTER);

        updateLocale();

        setVisible(true);
    }

    /**
     * DOCUMENT ME!
     *
     * @param event DOCUMENT ME!
     */
    public void controllerUpdate(ControllerEvent event) {
        if (event instanceof TimeEvent || event instanceof StopEvent) {
            updateLabel();
        }
    }

    /**
     * Update the time label.
     */
    public void updateLabel() {
        switch (mode) {
        case TC_PAL:
            timeLabel.setText(TimeFormatter.toTimecodePAL(getMediaTime()));

            break;

        case TC_NTSC:
            timeLabel.setText(TimeFormatter.toTimecodeNTSC(getMediaTime()));

            break;

        default:
            timeLabel.setText(TimeFormatter.toString(getMediaTime()));
        }

        repaint();
    }

    /**
     * Input box for setting the time where the crosshair should jump to
     */
    public void showCrosshairTimeInputBox() {
        String strNewTime;
        boolean bAgain = true;

        while (bAgain == true) {
            strNewTime = JOptionPane.showInputDialog(this,
                    STRCROSSHAIRTIMEINPUTBOX1, STRCROSSHAIRTIMEINPUTBOX2,
                    JOptionPane.PLAIN_MESSAGE);

            if ((strNewTime != null) && (!strNewTime.equals(""))) {
                long lngSeconds = TimeFormatter.toMilliSeconds(strNewTime);

                if (lngSeconds >= 0.0) {
                    setMediaTime(lngSeconds);
                    bAgain = false;
                } else {
                    JOptionPane.showMessageDialog(this,
                        STRCROSSHAIRTIMEINPUTBOX3 + "\n" +
                        STRCROSSHAIRTIMEINPUTBOX4, STRCROSSHAIRTIMEINPUTBOX5,
                        JOptionPane.ERROR_MESSAGE);
                    bAgain = true;
                }
            }

            //cancel is clicked
            if (strNewTime == null) {
                break;
            }
        }
    }

    private void createPopupMenu() {
        popup = new JPopupMenu();
        formatBG = new ButtonGroup();
        formatMenu = new JMenu();
        hhmmssmsMI = new JRadioButtonMenuItem();
        timecodePalMI = new JRadioButtonMenuItem();
        timecodeNtscMI = new JRadioButtonMenuItem();
        formatBG.add(hhmmssmsMI);
        formatBG.add(timecodePalMI);
        formatBG.add(timecodeNtscMI);
        hhmmssmsMI.setSelected(true);
        hhmmssmsMI.addActionListener(this);
        timecodePalMI.addActionListener(this);
        timecodeNtscMI.addActionListener(this);
        popup.add(formatMenu);
        formatMenu.add(hhmmssmsMI);
        formatMenu.add(timecodePalMI);
        formatMenu.add(timecodeNtscMI);
        updateLocale();
    }

    /**
     * DOCUMENT ME!
     */
    public void updateLocale() {
        STRCROSSHAIRTIMEINPUTBOX1 = ElanLocale.getString(
                "MediaPlayerControlPanel.STRCROSSHAIRTIMEINPUTBOX1");
        STRCROSSHAIRTIMEINPUTBOX2 = ElanLocale.getString(
                "MediaPlayerControlPanel.STRCROSSHAIRTIMEINPUTBOX2");
        STRCROSSHAIRTIMEINPUTBOX3 = ElanLocale.getString(
                "MediaPlayerControlPanel.STRCROSSHAIRTIMEINPUTBOX3");
        STRCROSSHAIRTIMEINPUTBOX4 = ElanLocale.getString(
                "MediaPlayerControlPanel.STRCROSSHAIRTIMEINPUTBOX4");
        STRCROSSHAIRTIMEINPUTBOX5 = ElanLocale.getString(
                "MediaPlayerControlPanel.STRCROSSHAIRTIMEINPUTBOX5");

        if (popup != null) {
            formatMenu.setText(ElanLocale.getString(
                    "TimeCodeFormat.Label.TimeFormat"));
            hhmmssmsMI.setText(ElanLocale.getString("TimeCodeFormat.TimeCode"));
            timecodePalMI.setText(ElanLocale.getString(
                    "TimeCodeFormat.TimeCode.SMPTE.PAL"));
            timecodeNtscMI.setText(ElanLocale.getString(
                    "TimeCodeFormat.TimeCode.SMPTE.NTSC"));
        }
    }

    /**
     * DOCUMENT ME!
     */
    public void updateActiveAnnotation() {
    }

    /**
     * DOCUMENT ME!
     */
    public void updateSelection() {
    }

    /* (non-Javadoc)
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
        if (timecodePalMI.isSelected()) {
            mode = TC_PAL;
        } else if (timecodeNtscMI.isSelected()) {
            mode = TC_NTSC;
        } else {
            mode = TC;
        }

        updateLabel();
    }

    /**
     * Handles a mouse click on the time label
     */
    private class TimeLabelMouseHandler extends MouseAdapter {
        /**
         * The user clicked on the time label so bring up an input box
         *
         * @param e DOCUMENT ME!
         */
        public void mouseClicked(MouseEvent e) {
            showCrosshairTimeInputBox();
        }

        public void mousePressed(MouseEvent e) {
            Point pp = e.getPoint();

            // HS nov 04: e.isPopupTrigger always returns false on my PC...
            if (SwingUtilities.isRightMouseButton(e) || e.isPopupTrigger()) {
                if (popup == null) {
                    createPopupMenu();
                }

                popup.show(TimePanel.this, pp.x, pp.y);
            }
        }
    }

    // end of TimeLabelMouseHandler
}
