/*
 * File:     TimeScale.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator;

import java.util.Vector;


/**
 *
 */
public class TimeScale {
    private Vector listeners;
    private long timeScaleBeginTime;
    private long timeScaleEndTime;
    private int timeScaleMsPerPixel;

    /**
     * Creates an empty TimeScale (begin time == end time).
     */
    public TimeScale() {
        listeners = new Vector();

        timeScaleBeginTime = 0;
        timeScaleEndTime = 0;
        timeScaleMsPerPixel = 10;
    }

    /**
     * Returns the begin time of the time scale in milli seconds.
     *
     * @return DOCUMENT ME!
     */
    public long getBeginTime() {
        return timeScaleBeginTime;
    }

    /**
     * Sets the beginTime of the time scale in milli seconds.
     *
     * @param beginTime DOCUMENT ME!
     */
    public void setBeginTime(long beginTime) {
        // Only update if needed.
        if (timeScaleBeginTime != beginTime) {
            timeScaleBeginTime = beginTime;

            // Tell all the interested TimeScalelisteners about the change
            notifyListeners();
        }
    }

    /**
     * Returns the end time of the time scale in milli seconds.
     *
     * @return DOCUMENT ME!
     */
    public long getEndTime() {
        return timeScaleEndTime;
    }

    /**
     * Sets the endTime of the time scale in milli seconds.
     *
     * @param endTime DOCUMENT ME!
     */
    public void setEndTime(long endTime) {
        // Only update if needed.
        if (timeScaleEndTime != endTime) {
            timeScaleEndTime = endTime;

            // Tell all the interested TimeScalelisteners about the change
            notifyListeners();
        }
    }

    /**
     * Returns the duration of the visible interval in the time scale in milli
     * seconds.
     *
     * @return DOCUMENT ME!
     */
    public long getIntervalDuration() {
        return timeScaleEndTime - timeScaleBeginTime;
    }

    /**
     * Returns the step size of the time scale in milli seconds.
     *
     * @return DOCUMENT ME!
     */
    public int getMsPerPixel() {
        return timeScaleMsPerPixel;
    }

    /**
     * Sets the step size of the time scale in milli seconds.
     *
     * @param msPerPixel DOCUMENT ME!
     */
    public void setMsPerPixel(int msPerPixel) {
        // Only update if needed.
        if (timeScaleMsPerPixel != msPerPixel) {
            timeScaleMsPerPixel = msPerPixel;

            // Tell all the interested TimeScalelisteners about the change
            notifyListeners();
        }
    }

    /**
     * Add a listener for TimeScale events.
     *
     * @param listener the listener that wants to be notified for DisplayState
     *        events.
     */
    public void addTimeScaleListener(TimeScaleListener listener) {
        listeners.add(listener);
        listener.updateTimeScale();
    }

    /**
     * Remove a listener for TimeScale events.
     *
     * @param listener the listener that no longer wants to be notified for
     *        TimeScale events.
     */
    public void removeTimeScaleListener(TimeScaleListener listener) {
        listeners.remove(listener);
    }

    /**
     * Tell all the listeners what the current TimeScale is.
     */
    public void notifyListeners() {
        for (int i = 0; i < listeners.size(); i++) {
            ((TimeScaleListener) listeners.elementAt(i)).updateTimeScale();
        }
    }
}
