/*
 * File:     AddTierCommand.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.server.corpora.clom.Tier;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;

import java.util.Iterator;
import java.util.Locale;
import java.util.Vector;


/**
 * A Command to add a tier to a transcription.
 *
 * @author Hennie Brugman
 * @version 1.2
 */
public class AddTierCommand implements UndoableCommand {
    private String commandName;
    private TierImpl tier = null;

    // receiver
    private Transcription transcription;

    /**
     * Creates a new AddTierCommand instance
     *
     * @param theName the name of the command
     */
    public AddTierCommand(String theName) {
        commandName = theName;
    }

    /**
     * Adds a tier to the transcription.
     *
     * @param receiver the transcription
     * @param arguments the arguments:  <ul><li>arg[0] = the tier name
     *        (String)</li> <li>arg[1] = the parent tier (Tier)</li>
     *        <li>arg[2] = the linguistic type name (String) </li> <li>arg[3]
     *        = the participant name (String)</li> <li>arg[4] = the default
     *        language (Locale)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        // receiver is Transcription
        transcription = (Transcription) receiver;

        // arguments
        String tierName = (String) arguments[0];
        Tier parentTier = (Tier) arguments[1];
        String lingType = (String) arguments[2];
        String participant = (String) arguments[3];
        Locale locale = (Locale) arguments[4];

        if (transcription != null) {
            tier = new TierImpl(parentTier, tierName, null, transcription, null);

            Vector types = transcription.getLinguisticTypes();
            LinguisticType t = null;
            Iterator typeIter = types.iterator();

            while (typeIter.hasNext()) {
                t = (LinguisticType) typeIter.next();

                if (t.getLinguisticTypeName().equals(lingType)) {
                    break;
                }
            }

            tier.setLinguisticType(t);
            tier.setParticipant(participant);
            tier.setDefaultLocale(locale);

            // transcription does not perform any checks..
            if (transcription.getTierWithId(tierName) == null) {
                transcription.addTier(tier);
            }
        }
    }

    /**
     * The undo action.
     */
    public void undo() {
        if (tier != null) {
            transcription.removeTier(tier);
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if (tier != null) {
            // transcription does not perform any checks..
            if (transcription.getTierWithId(tier.getName()) == null) {
                transcription.addTier(tier);
            }
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
