/*
 * File:     AnnotationBeforeCA.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ActiveAnnotationListener;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;


/**
 * A command action for creating a new annotation before a given annotation.
 *
 * @author Han Sloetjes
 */
public class AnnotationBeforeCA extends CommandAction
    implements ActiveAnnotationListener {
    private Annotation activeAnnotation;

    /**
     * Creates a new AnnotationBeforeCA instance
     *
     * @param viewerManager the ViewerManager
     */
    public AnnotationBeforeCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.NEW_ANNOTATION_BEFORE);
        viewerManager.connectListener(this);
        setEnabled(false);
    }

    /**
     * Creates a new <code>Command</code>.
     */
    protected void newCommand() {
        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.NEW_ANNOTATION_BEFORE);
    }

    /**
     * The receiver of this CommandAction is the TierImpl object on which the
     * new annotation should be created.
     *
     * @return the receiver
     */
    protected Object getReceiver() {
        //return viewerManager.getActiveAnnotation().getAnnotation().getTier;
        return activeAnnotation.getTier();
    }

    /**
     * Returns the arguments for the related Command.
     *
     * @return the arguments for the related Command
     */
    protected Object[] getArguments() {
        Object[] args = new Object[1];
        args[0] = activeAnnotation;

        return args;
    }

    /**
     * On a change of ActiveAnnotation perform a check to determine whether
     * this action should be enabled or disabled.<br>
     * This depends on the type of the annotation and the type of the Tier it
     * belongs to.
     *
     * @see ActiveAnnotationListener#updateActiveAnnotation()
     */
    public void updateActiveAnnotation() {
        activeAnnotation = vm.getActiveAnnotation().getAnnotation();
        checkState();
    }

    /**
     * Enables or disables this <code>Action</code> depending on the caracteristics
     * of the active annotation (and therefore the tier it is on), if any.
     */
    protected void checkState() {
        setEnabled(false);

        if (activeAnnotation == null) {
            return;
        }

        TierImpl tier = (TierImpl) activeAnnotation.getTier();

        Constraint c = tier.getLinguisticType().getConstraints();

        if ((c != null) && c.supportsInsertion()) {
            setEnabled(true);
        }
    }
}
