/*
 * File:     ModifyGraphicAnnotationCA.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ActiveAnnotationListener;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.SVGAlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * A command action for modifying a graphical annotation.
 *
 * @author Han Sloetjes
 */
public class ModifyGraphicAnnotationCA extends CommandAction
    implements ActiveAnnotationListener {
    private Annotation activeAnnotation;

    /**
     * Creates a new ModifyGraphicAnnotationCA instance
     *
     * @param viewerManager the ViewerManager
     */
    public ModifyGraphicAnnotationCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.MODIFY_GRAPHIC_ANNOTATION);

        viewerManager.connectListener(this);
        setEnabled(false);
    }

    /**
     * Creates a new Command.
     */
    protected void newCommand() {
        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.MODIFY_GRAPHIC_ANNOTATION_DLG);
    }

    /**
     * The receiver of this CommandAction is the Annotation that should be
     * modified.
     *
     * @return the receiver of this command
     */
    protected Object getReceiver() {
        return activeAnnotation;
    }

    /**
     * Returns the arguments for this command
     *
     * @return the arguments for this command
     */
    protected Object[] getArguments() {
        return new Object[] { vm.getTranscription() };
    }

    /**
     * On a change of ActiveAnnotation perform a check to determine whether
     * this action should be enabled or disabled.<br>
     *
     * @see ActiveAnnotationListener#updateActiveAnnotation()
     */
    public void updateActiveAnnotation() {
        activeAnnotation = vm.getActiveAnnotation().getAnnotation();
        checkState();
    }

    /**
     * Check whether or not this Action should be enabled given the current
     * active annotation.
     */
    protected void checkState() {
        setEnabled(false);

        if (activeAnnotation == null) {
            return;
        }

        TierImpl tier = (TierImpl) activeAnnotation.getTier();

        if (activeAnnotation instanceof SVGAlignableAnnotation &&
                tier.getLinguisticType().hasGraphicReferences()) {
            setEnabled(true);
        }
    }
}
