/*
 * File:     PlaybackVolumeToggleCA.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ViewerManager2;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;
import javax.swing.KeyStroke;


/**
 * A CommandAction that stores and sets the toggle value for the playback
 * volume shortcut.
 *
 * @author Han Sloetjes
 */
public class PlaybackVolumeToggleCA extends CommandAction {
    /** the default rate */
    private final float defaultVolume = 1.0F;

    /** the initial toggle value */
    private float toggleValue = 0.7F;

    //start with the default rate
    private boolean isToggled = false;

    /**
     * Creates a new PlaybackVolumeToggleCA instance
     *
     * @param viewerManager the viewer manager
     */
    public PlaybackVolumeToggleCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);

        putValue(Action.ACCELERATOR_KEY,
            KeyStroke.getKeyStroke(KeyEvent.VK_V,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        putValue(SHORT_DESCRIPTION,
            ElanLocale.getString(ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE +
                "ToolTip"));
        putValue(DEFAULT, "VOLUME_TOGGLE");

        //putValue(Action.NAME, "");
    }

    /**
     * Creates a new PlaybackVolumeToggleCommand. Every time newCommand is
     * called the rate value should be toggled.
     */
    protected void newCommand() {
        isToggled = !isToggled;

        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);
    }

    /**
     * Returns the reciever, the viewermanager. The command should set the rate
     * of the master media player  and update the rate slider.
     *
     * @return the viewermanager
     */
    protected Object getReceiver() {
        return vm;
    }

    /**
     * Returns the new volume.
     *
     * @return the new volume
     */
    protected Object[] getArguments() {
        Object[] args = new Object[1];

        if (isToggled) {
            args[0] = new Float(toggleValue);
        } else {
            args[0] = new Float(defaultVolume);
        }

        return args;
    }

    /**
     * Sets the toggle value. The value should be between  0 and 100,
     * inclusive.
     *
     * @param value the toggle value
     */
    public void setToggleValue(float value) {
        if ((value >= 0.0F) && (value <= 1.0F)) {
            toggleValue = value;
        }
    }

    /**
     * Returns the toggle value.
     *
     * @return the toggle value.
     */
    public float getToggleValue() {
        return toggleValue;
    }
}
