/*
 * File:     ReplaceCommand.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.search.result.model.ElanMatch;
import mpi.eudico.client.annotator.search.result.model.Replace;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import mpi.search.content.result.model.ContentResult;

import java.util.logging.Logger;


/**
 * A Command that tokenizes the contents of the annotations a source tier new
 * annotations on a destination tier.
 *
 * @author Han Sloetjes
 */
public class ReplaceCommand implements UndoableCommand {
    private static final Logger LOG = Logger.getLogger(ReplaceCommand.class.getName());
    private String commandName;

    // store state
    private TranscriptionImpl transcription;
    private String replaceString;
    private Annotation[] modifiedAnnotations;
    private String[] oldValues;

    /**
     * Creates a new TokenizeCommand instance.
     *
     * @param name
     *            the name of the command
     */
    public ReplaceCommand(String name) {
        commandName = name;
    }

    /**
     * Undo the changes made by this command.
     */
    public void undo() {
        transcription.setNotifying(false);

        for (int i = 0; i < modifiedAnnotations.length; i++) {
            String currentValue = modifiedAnnotations[i].getValue();
            modifiedAnnotations[i].setValue(oldValues[i]);
            oldValues[i] = currentValue;
        }

        transcription.setNotifying(true);
    }

    /**
     * Redo the changes made by this command.
     */
    public void redo() {
        undo();
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver
     *            the TranscriptionImpl
     * @param arguments
     *            the arguments:
     *            <ul>
     *            <li>arg[0] = Result
     *            <li>arg[1] = Replace String</li>
     *            </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (TranscriptionImpl) receiver;

        ContentResult result = (ContentResult) arguments[0];
        replaceString = (String) arguments[1];
        oldValues = new String[result.getMatchCount()];
        modifiedAnnotations = new Annotation[result.getMatchCount()];

        for (int i = 1; i <= result.getMatchCount(); i++) {
            ElanMatch match = (ElanMatch) result.getMatch(i);
            modifiedAnnotations[i - 1] = match.getAnnotation();
            oldValues[i - 1] = modifiedAnnotations[i - 1].getValue();
        }

        Replace.execute(result, replaceString, transcription);
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
