/*
 * File:     SaveSelectionAsEafCommand.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.export.ExportSelectionAsEAF;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;


/**
 * A Command that creates a new Transcription, copies CV's, LinguisticTypes,
 * Tiers and thw annotations within/overlapping the selected time interval.
 * All copied annotations are forced in to the interval.
 *
 * @author Han Sloetjes
 */
public class SaveSelectionAsEafCommand implements Command {
    private String commandName;

    /**
     * Creates a new CopyTierCommand instance
     *
     * @param theName the name of the command
     */
    public SaveSelectionAsEafCommand(String theName) {
        commandName = theName;
    }

    /**
     * Creates a new Transcription and adds copies of CV's, Linguistic Types, Tiers
     * and the annotations within the selection. Overlapping annotations will be
     * forced into the selectioninterval.
     * <b>Note: </b>it is assumed the types and order of the arguments
     * are correct.
     *
     * @param receiver the transcription
     * @param arguments the arguments: <ul><li>arg[0] = the selection begin time
     *        (Long)</li> <li>arg[1] the selection end time (Long)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        TranscriptionImpl transcription = (TranscriptionImpl) receiver;
        long beginTime = ((Long) arguments[0]).longValue();
        long endTime = ((Long) arguments[1]).longValue();

        // double check
        if (beginTime == endTime) {
            return;
        }

        new ExportSelectionAsEAF(transcription, beginTime, endTime);
    }

    /**
     * Returns the name of the command
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
