/*
 * File:     SetPlaybackToggleCommand.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.server.corpora.clom.Transcription;


/**
 * Actually changes the playback rate and volume toggle values.
 *
 * @author Han Sloetjes
 */
public class SetPlaybackToggleCommand implements UndoableCommand {
    private String commandName;
    private Transcription transcription;
    private float oldRate;
    private float newRate;
    private float oldVolume = -1;
    private float newVolume = -1;

    /**
     * Creates a new SetPlaybackToggleCommand instance
     *
     * @param theName the name of the command
     */
    public SetPlaybackToggleCommand(String theName) {
        commandName = theName;
    }

    /**
     * The undo action.
     */
    public void undo() {
        if (newRate > -1) {
            PlaybackRateToggleCA action = (PlaybackRateToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_RATE_TOGGLE);
            action.setToggleValue(oldRate);
        }

        if (newVolume > -1) {
            PlaybackVolumeToggleCA action = (PlaybackVolumeToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);
            action.setToggleValue(oldVolume);
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if (newRate > -1) {
            PlaybackRateToggleCA action = (PlaybackRateToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_RATE_TOGGLE);
            action.setToggleValue(newRate);
        }

        if (newVolume > -1) {
            PlaybackVolumeToggleCA action = (PlaybackVolumeToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);
            action.setToggleValue(newVolume);
        }
    }

    /**
     * Applies the new values to the relevant CommandActions. <b>Note: </b>it
     * is assumed the types and order of the arguments are correct.
     *
     * @param receiver the transcription
     * @param arguments the arguments: <ul><li>arg[0] = the new value for the
     *        playback rate toggle (Float)</li> <li>arg[1] = the new value for
     *        the playback volume toggle (Float)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (Transcription) receiver;

        if ((arguments[0] != null) && arguments[0] instanceof Float) {
            newRate = ((Float) arguments[0]).floatValue();

            PlaybackRateToggleCA action = (PlaybackRateToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_RATE_TOGGLE);
            action.setToggleValue(newRate);
        }

        if ((arguments[1] != null) && arguments[1] instanceof Float) {
            newVolume = ((Float) arguments[1]).floatValue();

            PlaybackVolumeToggleCA action = (PlaybackVolumeToggleCA) ELANCommandFactory.getCommandAction(transcription,
                    ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE);
            action.setToggleValue(newVolume);
        }
    }

    /**
     * Returns the name of the command
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
