/*
 * File:     MultiFileChooser.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.*;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.HeadlessException;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.io.File;

import javax.swing.ButtonGroup;
import javax.swing.DefaultListModel;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.WindowConstants;
import javax.swing.border.CompoundBorder;
import javax.swing.border.EmptyBorder;
import javax.swing.border.LineBorder;
import javax.swing.border.TitledBorder;
import javax.swing.filechooser.FileFilter;


/**
 * A custom Multiple File Chooser dialog that does not use the
 * <code>setAccessory</code>  mechanism of JFileChooser. This mechanism is not
 * L&F independent, forcing the Metal  L&F on all platforms. This chooser adds
 * a JFileChooser and other ui elements to a JComponent in what should  be a
 * L&F independent way.
 *
 * @author Han Sloetjes
 * @version 1.0 july 2005
 */
public class MultiFileChooser extends JComponent implements ActionListener {
    /** generic multiple file chooser */
    public static final int GENERIC = 0;

    /**
     * a multiple file chooser that let the user switch between media file
     * filtering   and template file filtering
     */
    public static final int MEDIA_TEMPLATE = 1;
    private JFileChooser chooser;
    private JList fileList;
    private DefaultListModel model;
    private JPanel selPanel;
    private JRadioButton mediaRB;
    private JRadioButton templateRB;
    private JButton okButton;
    private JButton cancelButton;
    private JButton copyButton;
    private JButton removeButton;
    private JButton upButton;
    private JButton downButton;
    private String dialogTitle;
    private JDialog dialog;
    private FileFilter mediaFilter;
    private FileFilter templateFilter;
    private int returnValue = JFileChooser.ERROR_OPTION;
    private int mode = MEDIA_TEMPLATE;

    /**
     * Creates a new MultiFileChooser instance with the  default selection
     * mode, <code>GENERIC</code>
     */
    public MultiFileChooser() {
        this(GENERIC);
    }

    /**
     * Creates a new MultiFileChooser instance
     *
     * @param mode the mode, either <code>GENERIC</code> or
     *        <code>MEDIA_TEMPLATE</code>
     */
    public MultiFileChooser(int mode) {
        if ((mode >= GENERIC) && (mode <= MEDIA_TEMPLATE)) {
            this.mode = mode;
        }

        initComponents();
    }

    private void initComponents() {
        mediaRB = new JRadioButton();
        templateRB = new JRadioButton();
        okButton = new JButton();
        cancelButton = new JButton();
        removeButton = new JButton();
        upButton = new JButton();
        downButton = new JButton();
        copyButton = new JButton();

        ImageIcon REMOVE_ICON = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/Remove.gif"));
        ImageIcon UP_ICON = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/Up.gif"));
        ImageIcon DOWN_ICON = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/Down.gif"));
        removeButton.setIcon(REMOVE_ICON);
        upButton.setIcon(UP_ICON);
        downButton.setIcon(DOWN_ICON);

        this.setLayout(new GridBagLayout());

        Insets insets = new Insets(2, 2, 2, 2);
        chooser = new JFileChooser();
        chooser.setMultiSelectionEnabled(true);
        chooser.setControlButtonsAreShown(false);
        chooser.setPreferredSize(new Dimension((int) chooser.getPreferredSize()
                                                            .getWidth() - 80,
                (int) chooser.getPreferredSize().getHeight()));

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        this.add(chooser, gbc);

        JPanel midPanel = new JPanel(new GridBagLayout());
        copyButton.setText(" >> ");
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.insets = new Insets(4, 0, 4, 6);
        midPanel.add(copyButton, gbc);

        selPanel = new JPanel(new GridBagLayout());
        selPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "Frame.ElanFrame.NewDialog.RadioFileType")));

        ButtonGroup bg = new ButtonGroup();
        mediaRB.setSelected(true);
        mediaRB.setText(ElanLocale.getString(
                "Frame.ElanFrame.NewDialog.RadioButtonMedia"));
        templateRB.setText(ElanLocale.getString(
                "Frame.ElanFrame.NewDialog.RadioButtonTemplate"));
        bg.add(mediaRB);
        bg.add(templateRB);
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        selPanel.add(mediaRB, gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.anchor = GridBagConstraints.NORTHWEST;
        gbc.insets = insets;
        selPanel.add(templateRB, gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.anchor = GridBagConstraints.SOUTH;
        gbc.insets = new Insets(4, 0, 4, 6);
        midPanel.add(selPanel, gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTH;

        //gbc.insets = insets;
        gbc.fill = GridBagConstraints.VERTICAL;
        gbc.weighty = 1.0;
        this.add(midPanel, gbc);

        JPanel rightPanel = new JPanel(new GridBagLayout());
        JPanel infoPanel = new JPanel(new GridBagLayout());
        Dimension dim = new Dimension(70, 40);
        infoPanel.setPreferredSize(dim);
        infoPanel.setMinimumSize(dim);
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weighty = 1.0;
        infoPanel.add(new JLabel(ElanLocale.getString(
                    "Frame.ElanFrame.NewDialog.Selected")));

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.insets = insets;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        rightPanel.add(infoPanel, gbc);

        model = new DefaultListModel();
        fileList = new JList(model);
        fileList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);

        JScrollPane jsp = new JScrollPane(fileList);
        jsp.setPreferredSize(new Dimension(jsp.getPreferredSize().getSize().width -
                30, jsp.getPreferredSize().getSize().height));

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.insets = insets;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        rightPanel.add(jsp, gbc);

        JPanel controlPanel = new JPanel(new GridLayout(1, 3, 6, 6));
        controlPanel.add(removeButton);
        controlPanel.add(upButton);
        controlPanel.add(downButton);

        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 2;
        gbc.insets = insets;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        rightPanel.add(controlPanel, gbc);

        gbc = new GridBagConstraints();
        gbc.gridx = 2;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.NORTH;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        this.add(rightPanel, gbc);

        JPanel buttonPanel = new JPanel(new GridLayout(1, 2, 6, 6));
        okButton.setText(ElanLocale.getString("Button.OK"));
        cancelButton.setText(ElanLocale.getString("Button.Cancel"));
        buttonPanel.add(okButton);
        buttonPanel.add(cancelButton);

        JPanel butBorderPanel = new JPanel(new GridBagLayout());
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.WEST;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        butBorderPanel.add(new JPanel(), gbc);
        gbc = new GridBagConstraints();
        gbc.gridx = 1;
        gbc.gridy = 0;
        gbc.anchor = GridBagConstraints.EAST;
        gbc.insets = insets;
        butBorderPanel.add(buttonPanel, gbc);

        butBorderPanel.setBorder(new CompoundBorder(
                new LineBorder(Color.GRAY, 1), new EmptyBorder(6, 6, 6, 6)));
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.gridy = 1;
        gbc.gridwidth = 3;
        gbc.anchor = GridBagConstraints.CENTER;
        gbc.insets = insets;
        gbc.fill = GridBagConstraints.HORIZONTAL;
        gbc.weightx = 1.0;
        this.add(butBorderPanel, gbc);

        chooser.addActionListener(this);
        copyButton.addActionListener(this);
        removeButton.addActionListener(this);
        upButton.addActionListener(this);
        downButton.addActionListener(this);
        okButton.addActionListener(this);
        cancelButton.addActionListener(this);
        mediaRB.addActionListener(this);
        templateRB.addActionListener(this);

        if (mode == GENERIC) {
            selPanel.setVisible(false);
        } else {
            //start with media file filter
            mediaFilter = ElanFileFilter.createFileFilter(ElanFileFilter.MEDIA_TYPE);
            templateFilter = ElanFileFilter.createFileFilter(ElanFileFilter.TEMPLATE_TYPE);
            chooser.addChoosableFileFilter(mediaFilter);

            String mediaDir = (String) Preferences.get("MediaDir", null);
            chooser.setCurrentDirectory(new File((mediaDir == null)
                    ? Constants.USERHOME : mediaDir));
        }
    }

    /**
     * Returns an array of file objects that the user has selected.
     *
     * @return an array of file objects
     */
    public Object[] getFiles() {
        Object[] obj = new Object[model.getSize()];

        for (int i = 0; i < obj.length; i++) {
            obj[i] = model.getElementAt(i);
        }

        return obj;
    }

    /**
     * Sets a list of files and/or directories in the multiple file box.
     *
     * @param files the files that should be added to the selected files list
     */
    public void setFiles(File[] files) {
        model.clear();

        for (int i = 0; i < files.length; i++) {
            model.insertElementAt(files[i], i);
        }

        fileList.setSelectedIndex(0);
        fileList.ensureIndexIsVisible(0);
    }

    /**
     * Clears the list of files.
     */
    public void clearList() {
        model.clear();
    }

    /**
     * Copies the selected files from the File chooser to the user list.
     */
    private void copyFile() {
        String strCurDir = null;

        if (chooser.isMultiSelectionEnabled()) {
            File[] files = chooser.getSelectedFiles();

            int curIndex;

            for (int i = 0; i < files.length; i++) {
                if (!model.contains(files[i])) {
                    curIndex = fileList.getSelectedIndex();
                    model.add(curIndex + 1, files[i]);
                    fileList.setSelectedIndex(curIndex + 1);
                }

                if (i == 0) {
                    strCurDir = files[i].getAbsolutePath();
                    strCurDir = new File(strCurDir).getParent();
                }
            }
        } else {
            File f = chooser.getSelectedFile();

            if (f != null) {
                int curIndex = fileList.getSelectedIndex();
                model.add(curIndex + 1, f);
                fileList.setSelectedIndex(curIndex + 1);

                strCurDir = f.getAbsolutePath();
                strCurDir = new File(strCurDir).getParent();
            }
        }

        // store dir
        boolean isMedia = false;
        boolean isTemplate = false;
        FileFilter[] filters = chooser.getChoosableFileFilters();

        for (int i = 0; i < filters.length; i++) {
            if (filters[i] == mediaFilter) {
                isMedia = true;

                break;
            }

            if (filters[i] == templateFilter) {
                isTemplate = true;

                break;
            }
        }

        if (isTemplate) {
            if (strCurDir != null) {
                Preferences.set("TemplateDir", strCurDir, null);
            }
        } else if (isMedia) {
            if (strCurDir != null) {
                Preferences.set("MediaDir", strCurDir, null);
            }
        }
    }

    /**
     * Removes the selected file from the selected files list.
     */
    private void removeFile() {
        int index = fileList.getSelectedIndex();

        if (index != -1) {
            model.removeElementAt(index);

            if (index >= model.getSize()) {
                index--;
            }

            fileList.setSelectedIndex(index);
            fileList.ensureIndexIsVisible(index);
        }
    }

    /**
     * Moves the selected file up in the list.
     */
    private void moveUp() {
        int index = fileList.getSelectedIndex();

        if (index > 0) {
            model.insertElementAt(model.remove(index), index - 1);

            fileList.setSelectedIndex(index - 1);
            fileList.ensureIndexIsVisible(index - 1);
        }
    }

    /**
     * Moves the selected file down in the list.
     */
    private void moveDown() {
        int index = fileList.getSelectedIndex();

        if ((index == -1) || (index == (model.getSize() - 1))) {
            return;
        }

        model.insertElementAt(model.remove(index), index + 1);

        fileList.setSelectedIndex(index + 1);
        fileList.ensureIndexIsVisible(index + 1);
    }

    /**
     * Sets the return value to <code>JFileChooser.APPROVE_OPTION</code> and
     * closes the dialog. Action after hitting the "OK" button.
     */
    private void apply() {
        returnValue = JFileChooser.APPROVE_OPTION;

        if (dialog != null) {
            dialog.dispose();
        }
    }

    /**
     * Sets the return value to <code>JFileChooser.CANCEL_OPTION</code> and
     * closes the dialog. Action after hitting the "Cancel" button.
     */
    private void cancel() {
        returnValue = JFileChooser.CANCEL_OPTION;

        if (dialog != null) {
            dialog.dispose();
        }
    }

    /**
     * Changes the active file filter to the media file filter (.mpg + .wav).
     */
    private void setMediaFilter() {
        boolean isMedia = false;
        FileFilter[] filters = chooser.getChoosableFileFilters();

        for (int i = 0; i < filters.length; i++) {
            if (filters[i] == mediaFilter) {
                isMedia = true;

                break;
            }
        }

        if (!isMedia) {
            File selectedFile = chooser.getSelectedFile();

            if (selectedFile != null) {
                String strTemplateDir = selectedFile.getAbsolutePath();
                strTemplateDir = (new File(strTemplateDir)).getParent();
                Preferences.set("TemplateDir", strTemplateDir, null);
            }

            chooser.removeChoosableFileFilter(templateFilter);
            chooser.addChoosableFileFilter(mediaFilter);

            String mediaDir = (String) Preferences.get("MediaDir", null);
            chooser.setCurrentDirectory(new File((mediaDir == null)
                    ? Constants.USERHOME : mediaDir));
        }
    }

    /**
     * Changes the active file filter to the template file filter (.etf).
     */
    private void setTemplateFilter() {
        boolean isTemplate = false;
        FileFilter[] filters = chooser.getChoosableFileFilters();

        for (int i = 0; i < filters.length; i++) {
            if (filters[i] == templateFilter) {
                isTemplate = true;

                break;
            }
        }

        if (!isTemplate) {
            File selectedFile = chooser.getSelectedFile();

            if (selectedFile != null) {
                String strMediaDir = selectedFile.getAbsolutePath();
                strMediaDir = (new File(strMediaDir)).getParent();
                Preferences.set("MediaDir", strMediaDir, null);
            }

            chooser.removeChoosableFileFilter(mediaFilter);
            chooser.addChoosableFileFilter(templateFilter);

            String templateDir = (String) Preferences.get("TemplateDir", null);
            chooser.setCurrentDirectory(new File((templateDir == null)
                    ? Constants.USERHOME : templateDir));
        }
    }

    /**
     * Action Listener implementation.
     *
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent e) {
        Object src = e.getSource();

        if (src == copyButton) {
            copyFile();
        } else if (src == removeButton) {
            removeFile();
        } else if (src == upButton) {
            moveUp();
        } else if (src == downButton) {
            moveDown();
        } else if (src == okButton) {
            apply();
        } else if (src == cancelButton) {
            cancel();
        } else if (src == chooser) {
            copyFile();
        } else if (src == mediaRB) {
            setMediaFilter();
        } else if (src == templateRB) {
            setTemplateFilter();
        }
    }

    /**
     * Main method for testing purposes.
     *
     * @param args app arguments
     */
    public static void main(String[] args) {
        /*
           try {
               UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
           } catch (Exception ex) {
               ex.printStackTrace();
           }
         */
        /*
        int v = new MultiFileChooser(MEDIA_TEMPLATE).showDialog(null, null);

        //System.out.println("Return: " + v);
        System.exit(0);
        */
    }

    /**
     * Copied from JFileChooser.
     *
     * @param parent the parent component, usually a frame
     * @param approveButtonText the text for the "OK" or approve button
     *
     * @return JFileChooser.ERROR_OPTION, APPROVE_OPTION or CANCEL_OPTION
     *
     * @throws HeadlessException
     *
     * @see JFileChooser.showDialog(Component, String)
     */
    public int showDialog(Component parent, String approveButtonText)
        throws HeadlessException {
        if (approveButtonText != null) {
            setApproveButtonText(approveButtonText);
        }

        dialog = createDialog(parent);
        dialog.addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    returnValue = JFileChooser.CANCEL_OPTION;
                }
            });
        returnValue = JFileChooser.ERROR_OPTION;

        dialog.show(); // blocks...

        dialog.dispose();
        dialog = null;

        return returnValue;
    }

    /**
     * Copied from JFileChooser. Creates a configured dialog.
     *
     * @param parent the parent
     *
     * @return a dialog object
     *
     * @throws HeadlessException
     */
    protected JDialog createDialog(Component parent) throws HeadlessException {
        Frame frame = (parent instanceof Frame) ? (Frame) parent
                                                : (Frame) SwingUtilities.getAncestorOfClass(Frame.class,
                parent);

        JDialog dialog = new JDialog(frame, dialogTitle, true);

        Container contentPane = dialog.getContentPane();
        contentPane.setLayout(new BorderLayout());
        contentPane.add(this, BorderLayout.CENTER);

        dialog.pack();
        dialog.setLocationRelativeTo(parent);

        return dialog;
    }

    /**
     * Returns the current mode.
     *
     * @return the current mode, either <code>GENERIC</code> or
     *         <code>MEDIA_TEMPLATE</code>
     */
    public int getMode() {
        return mode;
    }

    /**
     * Sets the mode for the chooser dialog.
     *
     * @param mode the new mode, either <code>GENERIC</code> or
     *        <code>MEDIA_TEMPLATE</code>
     */
    public void setMode(int mode) {
        if (this.mode != mode) {
            this.mode = mode;

            if (mode == GENERIC) {
                selPanel.setVisible(false);
            } else {
                selPanel.setVisible(true);
            }
        }
    }

    /**
     * Returns the current directory.
     *
     * @return the current directory
     */
    public File getCurrentDirectory() {
        return chooser.getCurrentDirectory();
    }

    /**
     * Sets the directory for the chooser.
     *
     * @param dir the directory for the chooser
     */
    public void setCurrentDirectory(File dir) {
        chooser.setCurrentDirectory(dir);
    }

    /**
     * Sets the text for the approve button.
     *
     * @param approveButtonText the text for the approve button
     */
    public void setApproveButtonText(String approveButtonText) {
        okButton.setText(approveButtonText);
    }

    /**
     * Returns the current text of the approve button.
     *
     * @return the current approve button text
     */
    public String getApproveButtonText() {
        return okButton.getText();
    }

    /**
     * Sets the file filter.
     *
     * @param filter the file filter for the chooser
     */
    public void setFileFilter(FileFilter filter) {
        chooser.setFileFilter(filter);
    }

    /**
     * Returns the current file filter
     *
     * @return the current file filter
     */
    public FileFilter getFileFilter() {
        return chooser.getFileFilter();
    }

    /**
     * Returns the set of file filters.
     *
     * @return the set of file filters
     */
    public FileFilter[] getChoosableFileFilters() {
        return chooser.getChoosableFileFilters();
    }

    /**
     * Adds a filter to the file filter.
     *
     * @param filter a file filter
     */
    public void addChoosableFileFilter(FileFilter filter) {
        chooser.addChoosableFileFilter(filter);
    }

    /**
     * Removes the specified filter from the set of filters
     *
     * @param f the filter to remove
     *
     * @return true if the filter was part of the set and successfully removed
     */
    public boolean removeChoosableFileFilter(FileFilter f) {
        return chooser.removeChoosableFileFilter(f);
    }

    /**
     * Removes all (choosable) file filters.
     */
    public void resetChoosableFileFilters() {
        chooser.resetChoosableFileFilters();
    }

    /**
     * Sets the file selection mode for the chooser!
     *
     * @param mode the file selection mode either JFileChooser.FILES_ONLY,
     *        JFileChooser.DIRECTORIES_ONLY or
     *        JFileChooser.FILES_AND_DIRECTORIES
     */
    public void setFileSelectionMode(int mode) {
        chooser.setFileSelectionMode(mode);
    }

    /**
     * Returns the file selection mode of the chooser
     *
     * @return the file selection mode
     *
     * @see #setFileSelectionMode(int)
     */
    public int getFileSelectionMode() {
        return chooser.getFileSelectionMode();
    }

    /**
     * Returns whether files can be selected.
     *
     * @return true if files can be selected in the chooser
     *
     * @see #setFileSelectionMode
     * @see #getFileSelectionMode
     */
    public boolean isFileSelectionEnabled() {
        return chooser.isFileSelectionEnabled();
    }

    /**
     * Sets the title for the dialog.
     *
     * @param title the title
     */
    public void setDialogTitle(String title) {
        dialogTitle = title;

        if (dialog != null) {
            dialog.setTitle(title);
        }
    }
}
