/*
 * File:     InterlinearAnnotation.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.interlinear;

import mpi.eudico.server.corpora.clomimpl.abstr.AbstractAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;
import mpi.eudico.server.corpora.clomimpl.type.LinguisticType;


/**
 * A class that holds the value of an annotation in one or more lines. In
 * addition it stores x position and size of the annotation.
 *
 * @author Han Sloetjes
 */
public class InterlinearAnnotation {
    /** type for a root, top level annotation */
    public static final int ROOT = 0;

    /** annotation of type association */
    public static final int ASSOCIATION = 1;

    /** annotation of type subdivision */
    public static final int SUBDIVISION = 2;

    /** virtual annotation of type timecode */
    public static final int TIMECODE = 3;
    private String value;
    private String[] lines;
    private String tierName;

    /** begin time */
    public long bt;

    /** end time */
    public long et;

    /** the x position */
    public int x;

    /** the y position */

    //public int y;

    /** the real width of the annotation */
    public int realWidth;

    /** the calculated width based on context (child / parent annotations) */
    public int calcWidth;

    /** the number of lines this annotation needs in case of wrapping */
    public int nrOfLines = 1;

    /** a field for the type of the annotation */
    public int type = ROOT;

    /**
     * Creates a new InterlinearAnnotation instance
     *
     * @param aa the source annotation
     */
    public InterlinearAnnotation(AbstractAnnotation aa) {
        if (aa != null) {
            value = aa.getValue().replace('\n', ' ').trim();
            bt = aa.getBeginTimeBoundary();
            et = aa.getEndTimeBoundary();

            TierImpl t = (TierImpl) aa.getTier();

            tierName = t.getName();

            if (t.hasParentTier()) {
                LinguisticType lt = t.getLinguisticType();

                if ((lt != null) && (lt.getConstraints() != null)) {
                    if (lt.getConstraints().getStereoType() == Constraint.SYMBOLIC_ASSOCIATION) {
                        type = ASSOCIATION;
                    } else {
                        type = SUBDIVISION;
                    }
                } else {
                    // something is wrong
                    type = ASSOCIATION;
                }
            }

            // default = root
        }
    }

    /**
     * Creates a new InterlinearAnnotation instance for a timecode value
     *
     * @param timecode the timecode value
     * @param tcTierName the (internal) identifier of the timecode tier
     */
    public InterlinearAnnotation(String timecode, String tcTierName) {
        value = timecode;
        type = 3;
        tierName = tcTierName;
    }

    /**
     * Returns the single value of the annotation.
     *
     * @return the single value of the annotation
     */
    public String getValue() {
        return value;
    }

    /**
     * Returns the lines of the annotation in case of wrapping.
     *
     * @return an array of substrings
     */
    public String[] getLines() {
        return lines;
    }

    /**
     * When the annotation value has to be wrapped, the resulting lines can be
     * passed  to this object. Wrapping is not done by this class itself; it
     * can depend on available number of pixels or characters, font size and
     * font metrics etc.
     *
     * @param lines the calculated substrings
     */
    public void setLines(String[] lines) {
        this.lines = lines;

        if (lines != null) {
            nrOfLines = lines.length;
        } else {
            nrOfLines = 1;
        }
    }

    /**
     * Returns the annotation's tier name.
     *
     * @return the annotation's tier name
     */
    public String getTierName() {
        return tierName;
    }

    /**
     * Returns the value of the annotation.
     *
     * @return the value of the annotation
     */
    public String toString() {
        return value;
    }
}
