/*
 * File:     PreviewPanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.interlinear;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.AdjustmentEvent;
import java.awt.event.AdjustmentListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;
import javax.swing.JScrollPane;


/**
 * Renders a print preview of interlinearized content.
 *
 * @author HS
 * @version 1.0
 */
public class PreviewPanel extends JPanel implements ComponentListener,
    AdjustmentListener {
    private Interlinear interlinear;
    private BufferedImage bi;
    private ImagePanel drawPanel;
    private JScrollPane scrollPane;
    private int[] offset = { 0, 0 };
    private Dimension visibleDimensions = new Dimension(550, 600);

    /**
     * Creates a new PreviewPanel instance
     *
     * @param interlinear the <code>Interlinear</code> object
     */
    public PreviewPanel(Interlinear interlinear) {
        this.interlinear = interlinear;
        initComponents();
    }

    /**
     * Initialises the user interface components.
     */
    private void initComponents() {
        drawPanel = new ImagePanel();

        //drawPanel.setDoubleBuffered(true);
        scrollPane = new JScrollPane(drawPanel);
        scrollPane.setPreferredSize(visibleDimensions);

        scrollPane.getHorizontalScrollBar().addAdjustmentListener(this);
        scrollPane.getVerticalScrollBar().addAdjustmentListener(this);

        //scrollPane.setDoubleBuffered(true);
        setLayout(new GridBagLayout());

        GridBagConstraints gbc = new GridBagConstraints();
        gbc.fill = GridBagConstraints.BOTH;
        gbc.weightx = 1.0;
        gbc.weighty = 1.0;
        add(scrollPane, gbc);

        //setBackground(Color.yellow);
        createBufferedImage(visibleDimensions.width, visibleDimensions.height);
        addComponentListener(this);
    }

    /**
     * Creates a BufferedImage of the given width and height.
     *
     * @param width the width of the buffer
     * @param height the height of the buffer
     */
    private void createBufferedImage(int width, int height) {
        if ((bi == null) || (bi.getWidth() < width) ||
                (bi.getHeight() < height)) {
            bi = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
            repaint();
        }
    }

    /**
     * Returns the BufferedImage used for the buffered rendering.
     *
     * @return the BufferedImage
     */
    public BufferedImage getBufferedImage() {
        return bi;
    }

    /**
     * Returns the current horizontal and vertical (scroll) offset.
     *
     * @return an array of size 2, the current horizontal and vertical (scroll)
     *         offset
     */
    public int[] getOffset() {
        return offset;
    }

    /**
     * Sets the size of the panel. This causes the scrollpane to update its
     * scrollbars.
     *
     * @param size the new size of the panel
     */
    public void setImageableSize(Dimension size) {
        drawPanel.setPreferredSize(size);
        drawPanel.revalidate();
    }

    /**
     * Causes the panel to adjust its size if necessary and the
     * interlinearization  to be repainted.
     */
    public void updateView() {
        drawPanel.setPreferredSize(new Dimension(interlinear.getWidth(),
                interlinear.getHeight()));
        drawPanel.revalidate();
        interlinear.drawViewOnImage(bi, offset);
        repaint();
    }

    /**
     * Implements the ComponentListener interface.  Invokes a repaint of the
     * linearization.
     *
     * @param e the component event
     */
    public void componentResized(ComponentEvent e) {
        createBufferedImage(this.getWidth(), this.getHeight());
        interlinear.drawViewOnImage(bi, offset);
        repaint();
    }

    /**
     * Implements the ComponentListener interface.
     *
     * @param e the component event
     */
    public void componentMoved(ComponentEvent e) {
    }

    /**
     * Implements the ComponentListener interface.
     *
     * @param e the component event
     */
    public void componentShown(ComponentEvent e) {
    }

    /**
     * Implements the ComponentListener interface.
     *
     * @param e the component event
     */
    public void componentHidden(ComponentEvent e) {
    }

    /**
     * Implements the AdjustmentListener interface.
     *
     * @param e the adjustment event
     */
    public void adjustmentValueChanged(AdjustmentEvent e) {
        offset[0] = scrollPane.getHorizontalScrollBar().getValue();
        offset[1] = scrollPane.getVerticalScrollBar().getValue();

        interlinear.drawViewOnImage(bi, offset);
        repaint();
    }

    /**
     * A class for painting a buffered image to its graphics environment
     *
     * @author HS
     * @version 1.0
     */
    private class ImagePanel extends JPanel {
        /**
         * Overrides the JComponent's paintComponent method.
         *
         * @param g the graphics context
         */
        public void paintComponent(Graphics g) {
            super.paintComponent(g);

            if (bi != null) {
                Graphics2D g2d = (Graphics2D) g;
                g2d.drawImage(bi, offset[0], offset[1], null);

                //g2d.drawImage(bi, 0, 0, null);
            }
        }
    }
}
