/*
 * File:     TimeSeriesChangeEvent.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.timeseries;

import java.util.EventObject;


/**
 * An event used to notify that (the configuration of) a track or
 * (the configuration of) a timeseries source has changed.
 *
 * @author Han Sloetjes
 */
public class TimeSeriesChangeEvent extends EventObject {
    // constants

    /** edit type add: an object has been added */
    public static final int ADD = 0;

    /** edit type change: an object has been changed */
    public static final int CHANGE = 1;

    /** edit type delete: an object has been deleted */
    public static final int DELETE = 2;

    // subtypes for changes

    /** identifies a change in a timeseries source (or its configuration) */
    public static final int TS_SOURCE = 100;

    /** a change in one or more tracks */
    public static final int TRACK = 101;
    private int editType = CHANGE;
    private int editSourceType = TS_SOURCE;

    /**
     * Constructor
     *
     * @param source the source object of the event
     */
    public TimeSeriesChangeEvent(Object source) {
        super(source);
    }

    /**
     * Creates a new TimeSeriesChangeEvent instance
     *
     * @param source the source
     * @param editType the edit type, ADD, CHANGE or DELETE
     */
    public TimeSeriesChangeEvent(Object source, int editType) {
        this(source);
        this.editType = editType;
    }

    /**
     * Creates a new TimeSeriesChangeEvent instance
     *
     * @param source the source
     * @param editType the edit type, ADD, CHANGE or DELETE
     * @param editSourceType the (sub)type of source that has been changed, TRACK or TS_SOURCE
     */
    public TimeSeriesChangeEvent(Object source, int editType, int editSourceType) {
        this(source, editType);
        this.editSourceType = editSourceType;
    }

    /**
     * Returns the type of source that has been changed, either TRACK or
     * TS_SOURCE. When a TimeSeries source has changed extracted tracks might
     * also have changed.
     *
     * @return the (sub)type of source that has been changed, TRACK or TS_SOURCE
     */
    public int getEditSourceType() {
        return editSourceType;
    }

    /**
     * Returns the type of edit.
     *
     * @return the type of edit, ADD, CHANGE or DELETE
     */
    public int getEditType() {
        return editType;
    }
}
