/*
 * File:     TSSourceConfiguration.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.timeseries.config;

import mpi.eudico.client.annotator.timeseries.TimeSeriesConstants;


/**
 * A configuration class for timeseries data files.
 * This class adds a few fields to the generic option
 * of setting and getting properties.
 * The source file has to be passed to the constructor.
 * A time offset for all tracks in the file can be specified, as well as
 * a default or preferred reader classname and the sample type.
 * Configuration objects of individual tracks can be added to this configuration
 * object.
 *
 * @author Han Sloetjes
 */
public class TSSourceConfiguration extends TSConfigurationImpl {
    private String source;
    private String providerClassName;
    private int timeOrigin = 0;
    private String sampleType = TimeSeriesConstants.UNKNOWN_RATE_TYPE;

    /**
     * Creates a new TSSourceConfiguration instance
     *
     * @param source the source file path as a string
     */
    public TSSourceConfiguration(String source) {
        super();
        this.source = source;
    }

    /**
     * Returns the service provider classname, if specified.
     *
     * @return the fully qualified service provider classname
     */
    public String getProviderClassName() {
        return providerClassName;
    }

    /**
     * Returns the sample type string.
     *
     * @return a string representation of the sample type, fixed rate, variable rate etc
     */
    public String getSampleType() {
        return sampleType;
    }

    /**
     * Returns the source file path.
     *
     * @return the path to the source file
     */
    public String getSource() {
        return source;
    }

    /**
     * Returns the time offset for tracks in this file.
     *
     * @return the time offset for tracks in this file
     */
    public int getTimeOrigin() {
        return timeOrigin;
    }

    /**
     * Sets the fully qualified class name of the service provider for
     * this source file.
     *
     * @param string the fully qualified class name of the service provider
     */
    public void setProviderClassName(String string) {
        providerClassName = string;
    }

    /**
     * Sets the sample type.
     *
     * @see TimeSeriesConstants
     * @param string the sample type as string
     */
    public void setSampleType(String string) {
        sampleType = string;
    }

    /**
     * Sets the time offset for the tracks in this file.
     *
     * @param offset the time offset in ms
     */
    public void setTimeOrigin(int offset) {
        timeOrigin = offset;
    }
}
