/*
 * File:     AnnotationDataRecord.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.util;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.TimeSlot;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;


/**
 * A class to store annotation data that are essential for the programmatic
 * re-creation of an annotation.
 *
 * @author Han Sloetjes
 */
public class AnnotationDataRecord {
    /** the name of the tier the annotation belongs to */
    private String tierName;

    /** the value of the annotation */
    private String value;

    /**
     * the value returned by annotation.getBeginTimeBoundary, can be an
     * interpolated value
     */
    private long beginTime;

    /**
     * the value returned by annotation.getEndTimeBoundary, can be an
     * interpolated value
     */
    private long endTime;

    /** the isTimeAligned value of the begin time TimeSlot */
    private boolean beginTimeAligned;

    /** the isTimeAligned value of the end time TimeSlot */
    private boolean endTimeAligned;

    /**
     * Creates an AnnotationData object from the specified Annotation.
     *
     * @param annotation the Annotation
     */
    public AnnotationDataRecord(Annotation annotation) {
        TimeSlot ts = null;

        if (annotation != null) {
            value = annotation.getValue();
            beginTime = annotation.getBeginTimeBoundary();
            endTime = annotation.getEndTimeBoundary();

            tierName = annotation.getTier().getName();

            if (annotation instanceof AlignableAnnotation) {
                ts = ((AlignableAnnotation) annotation).getBegin();
                beginTimeAligned = ts.isTimeAligned();
                ts = ((AlignableAnnotation) annotation).getEnd();
                endTimeAligned = ts.isTimeAligned();
            }
        }
    }

    /**
     * Returns the begin time of the annotation. <br>
     * This can be  an interpolated time value.
     *
     * @return the begin time
     */
    public long getBeginTime() {
        return beginTime;
    }

    /**
     * Returns true when the TimeSlot belonging to the begin boundary is  time
     * aligned. Only an AlignableAnnotation has a TimeSlot reference.
     *
     * @return true if the begin time TimeSlot is timealignable, false
     *         otherwise
     */
    public boolean isBeginTimeAligned() {
        return beginTimeAligned;
    }

    /**
     * Returns the end time of the annotation. <br>
     * This can be  an interpolated time value.
     *
     * @return the end time
     */
    public long getEndTime() {
        return endTime;
    }

    /**
     * Returns true when the TimeSlot belonging to the end boundary is  time
     * aligned. Only an AlignableAnnotation has a TimeSlot reference.
     *
     * @return true if the end time TimeSlot is timealignable, false otherwise
     */
    public boolean isEndTimeAligned() {
        return endTimeAligned;
    }

    /**
     * Returns the name of the tier this annotation belongs to.
     *
     * @return the tier name
     */
    public String getTierName() {
        return tierName;
    }

    /**
     * The text value of annotation.
     *
     * @return the text value of the annotation
     */
    public String getValue() {
        return value;
    }

    /**
     * Returns the value of the annotation.
     *
     * @return a String representation of this object; is the same as the value
     */
    public String toString() {
        return value;
    }

    public void setBeginTime(long beginTime) {
        this.beginTime = beginTime;
    }

    public void setEndTime(long endTime) {
        this.endTime = endTime;
    }
}
