/*
 * File:     FileUtility.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.client.annotator.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;


/**
 * A utility class that performs some Elan specific File operations.
 *
 * @author Han Sloetjes
 */
public class FileUtility {
    /**
     * Converts a path to a file URL string. Takes care of Samba related
     * problems file:///path works for all files except for samba file
     * systems, there we need file://machine/path, i.e. 2 slashes instead of 3
     * Does not support relative paths.
     *
     * @param path the path to convert
     *
     * @return a file url string
     */
    public static String pathToURLString(String path) {
        if (path == null) {
            return null;
        }

        // replace all back slashes by forward slashes
        path = path.replace('\\', '/');

        // remove leading slashes and count them
        int n = 0;

        while (path.charAt(0) == '/') {
            path = path.substring(1);
            n++;
        }

        // add the file:// or file:/// prefix
        if (n == 2) {
            return "file://" + path;
        } else {
            return "file:///" + path;
        }
    }

    /**
     * Method to compare the file names in two file/url paths without their
     * path and extensions.
     *
     * @param path1 first file path
     * @param path2 seconds file path
     *
     * @return boolean true if the file names without path and extension are
     *         the same
     */
    public static boolean sameNameIgnoreExtension(String path1, String path2) {
        // no name gives fals, two nulls are equal but have no name
        if ((path1 == null) || (path2 == null)) {
            return false;
        }

        path1 = path1.replace('\\', '/'); // give all paths the same delimiter, as URLs can now be compared

        String name1 = path1;
        int delimiterPos = path1.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name1 = path1.substring(delimiterPos + 1);
        }

        int extensionPos = name1.lastIndexOf('.');

        if (extensionPos >= 0) {
            name1 = name1.substring(0, extensionPos);
        }

        path2 = path2.replace('\\', '/');

        String name2 = path2;

        delimiterPos = path2.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name2 = path2.substring(delimiterPos + 1);
        }

        extensionPos = name2.lastIndexOf('.');

        if (extensionPos >= 0) {
            name2 = name2.substring(0, extensionPos);
        }

        return name1.equals(name2);
    }

    /**
     * Returns the file name from a file path.
     *
     * @param path the path
     *
     * @return the filename part of the path
     */
    public static String fileNameFromPath(final String path) {
        if (path == null) {
            return null;
        }

        String name = path.replace('\\', '/');
        int delimiterPos = name.lastIndexOf('/');

        if (delimiterPos >= 0) {
            name = name.substring(delimiterPos + 1);
        }

        return name;
    }

    /**
     * Tests wether a file exists.
     *
     * @param path the filepath to test
     *
     * @return true if the file exists, false otherwise
     */
    public static boolean fileExists(final String path) {
        if (path == null) {
            return false;
        }

        // remove the file: part of the URL, leading slashes are no problem
        int colonPos = path.indexOf(':');
        String fileName = path;

        if (colonPos > -1) {
            fileName = path.substring(colonPos + 1);
        }

        // replace all back slashes by forward slashes
        fileName = fileName.replace('\\', '/');

        File file = new File(fileName);

        if (!file.exists()) {
            return false;
        } else {
            return true;
        }
    }

    //////////////////////////////////////
    // Copied from 'old' FileUtil  class
    //////////////////////////////////////

    /**
     * If file f is a file and has an extension, it is returned. Otherwise,
     * null is returned.
     *
     * @param f a File
     *
     * @return the file extension or null
     */
    public static final String getExtenstion(final File f) {
        String name = f.getName();
        int li = name.lastIndexOf(".");

        return (li == -1) ? null : name.substring(li + 1);
    }

    /**
     * Copies a file to a destination directory.
     *
     * @param sourceFile the source
     * @param destDir the destination directory
     *
     * @throws Exception DOCUMENT ME!
     */
    public void copyToDir(final File sourceFile, final File destDir)
        throws Exception {
        // this should better not be static!
        byte[] buffer = new byte[4096]; // You can change the size of this if you want.
        destDir.mkdirs(); // creates the directory if it doesn't already exist.

        File destFile = new File(destDir, sourceFile.getName());
        FileInputStream in = new FileInputStream(sourceFile);
        FileOutputStream out = new FileOutputStream(destFile);
        int len;

        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }

        out.close();
        in.close();
    }

    /**
     * Copies a source file to a destination file.
     *
     * @param sourceFile the source file
     * @param destFile the destination file
     *
     * @throws Exception DOCUMENT ME!
     */
    public static void copyToFile(final File sourceFile, final File destFile)
        throws Exception {
        // this should better not be static!
        byte[] buffer = new byte[4096]; // You can change the size of this if you want.
        FileInputStream in = new FileInputStream(sourceFile);
        FileOutputStream out = new FileOutputStream(destFile);
        int len;

        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }

        out.close();
        in.close();
    }
}
