/*
 * File:     CodeGroup.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clom;

import mpi.eudico.server.corpora.util.LabelNotUniqueException;

import java.util.Vector;


/**
 * A CodeGroup describes the legal values for all tracks in the
 * Tier's Tags. Each track's contents are described by a specific
 * CodeType. A CodeType also specifies unique labels for each
 * track and the order of the tracks.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 5-Jun-1998
 */
public interface CodeGroup {
    /**
     * Returns a list of label Strings, in the correct order. Tags
     * contain a list of values for each field in exactly the same
     * order.
     *
     * @return    the Labels of the CodeGroup
     */
    public Vector getLabels();

    /**
     * Gives the CodeType of the field associated with a specific
     * label String. This label is supposed to be unique for a given
     * CodeGroup.
     *
     * @param theLabel    label string for which to find the CodeType.
     * @return        CodeType associated with theLabel.
     */
    public CodeType getCodeType(String theLabel);

    /**
     * Adds a new track to the CodeGroup. A track is specified by a unique
     * label string and it's associated CodeType.
     *
     * @param theLabel    the track's unique label string
     * @param theCodeType    a CodeType
     */
    public void addTrack(String theLabel, CodeType theCodeType)
        throws LabelNotUniqueException;

    /**
     * Adds a new track to the CodeGroup at a specified position. A track is specified by a unique
     * label string and it's associated CodeType.
     *
     * @param theLabel    the track's unique label string
     * @param theCodeType    a CodeType
     * @param index        position where track is to be inserted in the CodeGroup
     */
    public void addTrackAt(String theLabel, CodeType theCodeType, int index)
        throws LabelNotUniqueException;

    /**
     * Returns true if this CodeGroup has a Tuple structure
     *
     * @return    true if the CodeGroup has a Tuple structure
     */
    public boolean isTuple();
}
