/*
 * File:     CodeType.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clom;

import mpi.eudico.server.corpora.util.SharedDataObject;
import mpi.eudico.server.corpora.util.ValuesNotListableException;

import java.util.Set;


/**
 * A CodeType describes legal values for a specific field in a Tag.
 * CodeTypes can be bundled to CodeGroups, that can be associated with
 * a set of Tiers.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 11-Sep-1998
 */
public interface CodeType extends SharedDataObject {
    public static final String FREE_TEXT = "free_text";
    public static final String CLASSIFIERS = "classifiers";

    /**
     * Gives the name of the CodeType.
     *
     * @return    the CodeType's name
     */
    public String getName();

    /**
     * Returns a string specifying if and how legal values are specified by this
     * CodeType. Example values are FREE_TEXT, CLASSIFIERS, etc.
     *
     * @return    the type of legal values returned by the CodeType
     */
    public String getValueType();

    /**
     * Gives a list of all values that are legal for a Tag field of this CodeType.
     * For several field types these values can not be listed, for example for free
     * text fields or (recursive) syntactic tree node specifications. In these cases
     * a ValuesNotListableException will be thrown.
     *
     * @return    a list of all legal values for a Tag field
     * @exception ValuesNotListableException    thrown when it is impossible to generate
     *                        a complete list of legal values
     */
    public Set getLegalValues() throws ValuesNotListableException;

    /**
     * Returns true if a specific string is considered legal by the CodeType.
     *
     * @return    true is s is legal
     */
    public boolean isLegalValue(String s);
}
