/*
 * File:     MetaTime.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clom;

import mpi.eudico.server.corpora.util.RemoteListIterator;


/**
 * MetaTime encapsulates the ordering of Tags (of multiple Tiers) in a
 * Transcription. It is considered to be part of the Transcription. Ordering
 * is only stored for time segments that are not aligned with media time.
 * So, the more tags are aligned with media time, the smaller the resulting
 * MetaTime is.
 * The MetaTime is used when comparing Tags in the Tag's compareTo method.
 * Given a constructed MetaTime, it is then sufficient to add Tags to a TreeSet,
 * they will be ordered according to the MetaTime automatically.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 20-Apr-1999
 */
public interface MetaTime extends RemoteListIterator {
    /**
     * Adds a Tag to the MetaTime at currentTag. The Tag
     * can be either time-aligned or not time-aligned. insertTag may only
     * be called from Tier.addTag
     *
     * @param theTag    the Tag to be inserted.
     */
    public void insertTag(Tag theTag);

    /**
     * A utility method to print the current state of MetaTime to standard output.
     */
    public void printMetaTime();

    /**
     * Returns true if tag1 starts before tag2, according to the order
     * specified by the MetaTime. Each Tag can be either time-aligned or
     * not time-aligned.
     *
     * @param tag1    first tag to be compared.
     * @param tag2    second tag to be compared.
     * @return    true if tag1 starts before tag2.
     */
    public boolean startsBefore(Tag tag1, Tag tag2);

    /**
     * Returns the number of elements in MetaTime.
     */
    public int size();
}
