/*
 * File:     CodeGroupImpl.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clomimpl.abstr;

import mpi.eudico.server.corpora.clom.CodeGroup;
import mpi.eudico.server.corpora.clom.CodeType;

import mpi.eudico.server.corpora.util.LabelNotUniqueException;

import java.util.HashMap;
import java.util.Set;
import java.util.Vector;


/**
 * A CodeGroup describes the legal values for all fields in the Tier's Tags.
 * Each field's contents are described by a specific CodeType.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 4-May-1999
 */
public class CodeGroupImpl extends CodeTypeImpl implements CodeGroup {
    /** Holds value of property DOCUMENT ME! */
    protected Vector labels;

    /** Holds value of property DOCUMENT ME! */
    protected HashMap codeTypes;

    /** Holds value of property DOCUMENT ME! */
    protected boolean isTuple;

    /**
     * Creates a new CodeGroupImpl instance
     */
    public CodeGroupImpl() {
        // The super class CodeTypeImpl has a 
        // constructor CodeTypeImpl(String theName, Set theValues)
        // All these values are unknown inside this constructor.
        super("", null);
        labels = new Vector();
        codeTypes = new HashMap();
        isTuple = false;
    }

    /**
     * Constructor with a flag that indicates whether the CodeGroup has a Tuple
     * structure
     *
     * @param isTuple flag to indicate Tuple structure
     */
    public CodeGroupImpl(boolean isTuple) {
        // The super class CodeTypeImpl has a 
        // constructor CodeTypeImpl(String theName, Set theValues)
        // All these values are unknown inside this constructor.
        super("", null);
        labels = new Vector();
        codeTypes = new HashMap();
        isTuple = isTuple;
    }

    /**
     * Constructor with all the parameters for the CodeTypeImpl constructor
     *
     * @param theName the name of the CodeType (CodeGroup).
     * @param theValues the legal values of the CodeType (CodeGroup).
     * @param isTuple flag to indicate Tuple structure
     */
    public CodeGroupImpl(String theName, Set theValues, boolean isTuple) {
        super(theName, theValues);
        labels = new Vector();
        codeTypes = new HashMap();
        isTuple = isTuple;
    }

    // CodeGroup interface methods

    /**
     * Returns a list of label Strings, in the correct order. Tags contain a
     * list of values for each field in exactly the same order.
     *
     * @return the Labels of the CodeGroup
     */
    public Vector getLabels() {
        return labels;
    }

    /**
     * Gives the CodeType of the field associated with a specific label String.
     * This label is supposed to be unique for a given CodeGroup.
     *
     * @param theLabel label string for which to find the CodeType.
     *
     * @return CodeType associated with theLabel.
     */
    public CodeType getCodeType(String theLabel) {
        return (CodeType) codeTypes.get(theLabel);
    }

    /**
     * Adds a new track to the CodeGroup. A track is specified by a unique
     * label string and it's associated CodeType.
     *
     * @param theLabel the track's unique label string
     * @param theCodeType a CodeType
     *
     * @throws LabelNotUniqueException DOCUMENT ME!
     */
    public void addTrack(String theLabel, CodeType theCodeType)
        throws LabelNotUniqueException {
        if (labels.contains(theLabel)) {
            throw new LabelNotUniqueException();
        } else {
            labels.addElement(theLabel);
            codeTypes.put(theLabel, theCodeType);
        }
    }

    /**
     * Adds a new track to the CodeGroup at a specified position.
     *
     * @param theLabel the track's unique label string
     * @param theCodeType a CodeType
     * @param index position where to insert the CodeType in the CodeGroup
     *
     * @throws LabelNotUniqueException DOCUMENT ME!
     */
    public void addTrackAt(String theLabel, CodeType theCodeType, int index)
        throws LabelNotUniqueException {
        if (labels.contains(theLabel)) {
            throw new LabelNotUniqueException();
        } else {
            labels.insertElementAt(theLabel, index);
            codeTypes.put(theLabel, theCodeType);
        }
    }

    /**
     * Returns true if this CodeType has a Tuple structure
     *
     * @return true if the CodeType has a Tuple structure
     *
     */
    public boolean isTuple() {
        return isTuple;
    }
}
