/*
 * File:     CodeTypeImpl.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clomimpl.abstr;

import mpi.eudico.server.corpora.clom.CodeType;

import mpi.eudico.server.corpora.util.ValuesNotListableException;

import java.util.Set;


/**
 * A CodeType describes legal values for a specific field in a Tag. CodeTypes
 * can be bundled to CodeGroups, that can be associated with a set of Tiers.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 4-May-1999
 * @version Aug 2005 Identity removed
 */
public class CodeTypeImpl implements CodeType {
    /** The name of the CodeType. */
    private String codeTypeName;

    /** The owner of the CodeType. */
    protected String owner;

    /**
     * Legal values associated with the CodeType. Is null for free text
     * CodeTypes
     */
    private Set legalValues;

    /**
     * Creates a new CodeTypeImpl instance
     *
     * @param theName DOCUMENT ME!
     * @param theValues DOCUMENT ME!
     */
    public CodeTypeImpl(String theName, Set theValues) {
        codeTypeName = theName;
        legalValues = theValues;
    }

    // CodeType interface methods

    /**
     * Gives the name of the CodeType.
     *
     * @return the CodeType's name
     *
     */
    public String getName() {
        return codeTypeName;
    }

    /**
     * Returns a string specifying if and how legal values are specified by
     * this CodeType. Example values are FREE_TEXT, CLASSIFIERS, etc.
     *
     * @return the type of legal values returned by the CodeType
     *
     */
    public String getValueType() {
        if (legalValues == null) {
            return FREE_TEXT;
        } else {
            return CLASSIFIERS;
        }
    }

    /**
     * Gives a list of all values that are legal for a Tag field of this
     * CodeType. For several field types these values can not be listed, for
     * example for free text fields or (recursive) syntactic tree node
     * specifications. In these cases a ValuesUndeterminedException will be
     * thrown. In case of valueType FREE_TEXT null is returned.
     *
     * @return a list of all legal values for a Tag field
     *
     * @throws ValuesNotListableException DOCUMENT ME!
     */
    public Set getLegalValues() throws ValuesNotListableException {
        return legalValues;
    }

    /**
     * Returns true if a specific string is considered legal by the CodeType.
     *
     * @param s DOCUMENT ME!
     *
     * @return true is s is legal
     *
     */
    public boolean isLegalValue(String s) {
        return legalValues.contains(s);
    }

    // SharedDataObject interface method(s), via CodeType interface
}
