/*
 * File:     CorpusImpl.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clomimpl.abstr;

import mpi.eudico.server.corpora.clom.LeafCorpus;
import mpi.eudico.server.corpora.clom.Session;

import mpi.eudico.server.corpora.location.LocatorManager;

import mpi.eudico.server.corpora.util.DataTreeNode;

import java.util.Iterator;
import java.util.Vector;


/**
 * The CorpusImpl class is a LeafCorpus implementation class.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 10-Jun-1999 modified 2 oct 2000; Daan Broeder Corpus points now to sessions, not to transcriptions
 * @version Aug 2005 Identity removed
 */
public abstract class CorpusImpl implements LeafCorpus {
    /** The name of the Corpus. */
    private String name;

    /**
     * The list of Session objects that are momentarily in use by some client.
     */
    protected Vector sessions;

    /** The list of MetaData prtaining to this corpus */
    protected Vector metaData;

    /**
     * Parent in the Corpus hierarchy. This can be either a CompositeCorpus or
     * the CorpusDatabase.
     */
    protected DataTreeNode parent; // back reference
    protected LocatorManager locatorManager;

    /**
     * Constructor for Leaf Corpora that have a parent not equal to the
     * CorpusDatabase
     *
     * @param theName DOCUMENT ME!
     * @param theLocatorMgr DOCUMENT ME!
     */
    public CorpusImpl(String theName, LocatorManager theLocatorMgr) {
        name = theName;

        locatorManager = theLocatorMgr;

        sessions = new Vector();
    }

    /**
     * Constructor for Leaf Corpora that represent the top level in a Corpus
     * hierarchy
     *
     * @param theName the name of this Corpus
     */
    public CorpusImpl(String theName) {
        name = theName;

        sessions = new Vector();
    }

    // TreeViewable interface methods

    /**
     * Gives the name of the leaf Corpus.
     *
     * @return the name of the Corpus
     */
    public String getNodeName() {
        return getName();
    }

    /**
     * Corpus is never a Leaf in the browsable Corpus tree (since it contains
     * sessions as leafs), and therefore always returns false.
     *
     * @return false
     */
    public boolean isTreeViewableLeaf() {
        return false;
    }

    /**
     * Gives all children of this node in the tree of corpora, in this case all
     * Sessions that are accessible.
     *
     * @return a list of child objects, in this case CHAT Sessions
     */
    public Vector getChildren() {
        return getSessions();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Vector getMetaData() {
        return null;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean hasMetaData() {
        return false;
    }

    // LeafCorpus interface methods

    /**
     * Returns the name of the Corpus.
     *
     * @return name of Corpus
     */
    public String getName() {
        return name;
    }

    /**
     * Adds a reference to a Session in the sessions list.
     *
     * @param theSession the Session who's reference is to be added
     */
    public final void addSession(Session theSession) {
        sessions.add(theSession);
    }

    /**
     * Removes the reference to a Session from the sessions list. The garbage
     * collector will then actually remove the Session object.
     *
     * @param theSession the Session who's reference is to be removed
     */
    public void removeSession(Session theSession) {
        sessions.remove(theSession);
    }

    /**
     * Prints out what items and how many of them are contained in this Corpus
     * to the EUDICO server's console.
     */
    public void printStatistics() {
        System.out.println("");
        System.out.println("Corpus name: " + name);

        System.out.println("-->Total number of sessions: " + sessions.size());

        Iterator iter = sessions.iterator();

        while (iter.hasNext()) {
            ((Session) iter.next()).printStatistics();
        }
    }

    // SharedDataObject interface method(s), via Corpus interface
    // Unreferenced interface methods

    /**
     * Is called (after an unpredictable time interval) when there are no more
     * remote references to an object of this class. It then removes all
     * references to component Sessions, that will then be deleted by the
     * garbage collector. The Corpus instance itself is not deleted, because
     * it may receive new requests from other clients.
     */
    public void unreferenced() {
        // remove all elements from sessions vector
        sessions.clear();
    }

    /**
     * Give all Tools that are applicable to a data object in the Corpus
     * hierarchy.
     *
     * @return the set of available tools
     */
    public Vector getAvailableTools() {
        return ToolDatabaseImpl.Instance().getAvailableTools(this);
    }

    /**
     * Removes a child in the Corpus data hierarchy by deleting the reference
     * to the child. The garbage collector will then do the actual deletion.
     * Children for CHATCorpus are CHAT Sessions.
     *
     * @param theChild the child to be deleted
     */
    public void removeChild(DataTreeNode theChild) {
        removeSession((Session) theChild);
    }
}
