/*
 * File:     SessionImpl.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clomimpl.abstr;

import mpi.eudico.server.corpora.clom.Session;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.location.LocatorManager;

import mpi.eudico.server.corpora.util.DataTreeNode;

import mpi.eudico.tool.ToolCondition;

import java.util.Iterator;
import java.util.Vector;


/**
 * SessionImpl implements Session.
 *
 * @author Daan Broeder
 * @version 27-Sept-2000
 * @version Aug 2005 Identity removed
 */
public abstract class SessionImpl implements Session {
    /**
     * The list of Transcriptions in this Session. transcriptions are
     * languageResources they are counted seperately because of the
     * (temporary) dependence of MediaObjects on Transcription
     */
    protected Vector transcriptions;

    /** The list of LanguageResources in this Session. */
    protected Vector languageResources;

    /** The meta-data associated with this Session */
    protected Vector metaData;

    /** Session name */
    protected String name;

    /** Holds value of property DOCUMENT ME! */
    protected String owner;
    private DataTreeNode parent; // back reference, used for deletion of Sessions

    /** Holds value of property DOCUMENT ME! */
    protected LocatorManager locatorManager;

    /**
     * Creates a new SessionImpl instance
     *
     * @param theName DOCUMENT ME!
     * @param theParent DOCUMENT ME!
     * @param theLocatorMgr DOCUMENT ME!
     */
    public SessionImpl(String theName, DataTreeNode theParent,
        LocatorManager theLocatorMgr) {
        name = theName;
        parent = theParent;
        locatorManager = theLocatorMgr;

        transcriptions = new Vector();
        languageResources = new Vector();
        metaData = new Vector();
    }

    /**
     * Creates a new SessionImpl instance
     *
     * @param theName DOCUMENT ME!
     * @param theParent DOCUMENT ME!
     */
    public SessionImpl(String theName, DataTreeNode theParent) {
        name = theName;
        parent = theParent;

        transcriptions = new Vector();
        languageResources = new Vector();
        metaData = new Vector();
    }

    /**
     * Returns the name of the Session
     *
     * @return name of Session
     */
    public String getName() {
        return name;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public String getOwner() {
        return owner;
    }

    /**
     * Returns a list of LanguageResources that are accessible.

     *
     * @return DOCUMENT ME!
     */
    public Vector getLanguageResources() {
        // temporary hack it assumes that all media files are accessible through the getMediaObj() method
        // of Transcript
        System.out.println("getLanguageResource called for: " + getName());

        if (languageResources.size() == 0) {
            languageResources = getTranscriptions();

            for (int i = 0; i < languageResources.size(); i++) {
                try {
                    languageResources.add(((Transcription) transcriptions.get(i)).getMediaObject());
                } catch (Exception e) {
                    ;
                }
            }
        }

        return languageResources;
    }

    /**
     * does the Session hold LanguageResources
     *
     *
     * @return DOCUMENT ME!
     */
    public boolean hasLanguageResources() {
        if ((languageResources != null) && (languageResources.size() > 0)) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * Returns a list of Transcriptions that are accessible.
     *
     * @return DOCUMENT ME!
     */
    public abstract Vector getTranscriptions();

    /**
     * Returns a list of MetaData that are accessible.
     *
     * @return DOCUMENT ME!
     */
    public Vector getMetaData() {
        return null;
    }

    /**
     * does the Session have any MetaData
     *
     * @return DOCUMENT ME!
     */
    public boolean hasMetaData() {
        return false;
    }

    /**
     * DOCUMENT ME!
     *
     * @param tr DOCUMENT ME!
     */
    public void addTranscription(Transcription tr) {
        transcriptions.add(tr);
    }

    // TreeViewable interface methods
    public String getNodeName() {
        return getName();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean isTreeViewableLeaf() {
        return false;
    }

    //DGB; check first if getTranscriptions return != null
    public Vector getChildren() {
        Vector children = new Vector();

        if (getTranscriptions() != null) {
            children.addAll(getTranscriptions());
        }

        if (hasLanguageResources()) {
            children.addAll(getLanguageResources());
        }

        if (hasMetaData()) {
            children.addAll(getMetaData());
        }

        return children;
    }

    /**
     *
     *
     *
     */

    // SharedDataObject interface method(s), via Transcription interface
    // Unreferenced interface method
    public void unreferenced() {
        // corpus should store the only reference to session, so
        // removing this reference results in deletion by GC
        getParent().removeChild(this);
    }

    // ToolAdministrator interface method
    public Vector getAvailableTools() {
        return ToolDatabaseImpl.Instance().getAvailableTools(this);
    }

    /**
     * Returns true if this ToolAdministrator meets the condition specified in
     * the argument.
     *
     * @return DOCUMENT ME!
     */
    public abstract boolean checkToolCondition(ToolCondition theCondition);

    // DataTreeNode interface method

    /**
     * Returns the parent object in the hierarchy of Corpus data objects.
     *
     * @return the parent DataTreeNode
     */
    public DataTreeNode getParent() {
        return parent;
    }

    /**
     * Removes a child in the Corpus data hierarchy by deleting the reference
     * to the child. The garbage collector will then do the actual deletion.
     * removeChild is not yet implemented, therefore removeChild does nothing
     * yet.
     *
     * @param theChild the child to be deleted
     */
    public void removeChild(DataTreeNode theChild) {
        removeTranscription((Transcription) theChild);
    }

    /**
     * DOCUMENT ME!
     *
     * @param theTranscription DOCUMENT ME!
     */
    public void removeTranscription(Transcription theTranscription) {
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Object getConcreteData() {
        return null;
    }

    /**
     * Prints out what items and how many of them are contained in this Corpus
     * to the EUDICO server's console.
     */
    public void printStatistics() {
        System.out.println("");
        System.out.println("Session name: " + name);

        System.out.println("-->Total number of transcriptions: " +
            transcriptions.size());

        Iterator iter = transcriptions.iterator();

        while (iter.hasNext()) {
            ((Transcription) iter.next()).printStatistics();
        }
    }
}
