/*
 * File:     CVEntry.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.server.corpora.clomimpl.type;


/**
 * An entry in a ContolledVocabulary.<br>
 * An entry has a value and an optinal description.  Pending: the entries'
 * value in a controlled vocabulary should be unique. We could override the
 * equals(o) method of <code>Object</code> to return
 * this.value.equals(((CVEntry)o).getValue()). This however would not be
 * consistent with hashCode().
 *
 * @author Han Sloetjes
 * @version jun 04
 * @version apr 05, added equals() method
 */
public class CVEntry {
    private String value;
    private String description;

    /**
     * Creates a new entry with the specified value.
     *
     * @param value the value
     *
     * @see CVEntry(String, String)
     */
    public CVEntry(String value) {
        this(value, null);
    }

    /**
     * Creates a new entry with the specified value and the specified
     * description.
     *
     * @param value the value
     * @param description the description
     *
     * @exception IllegalArgumentException when the value is <code>null</code>
     *            or of length 0
     */
    public CVEntry(String value, String description) {
        if ((value == null) || (value.length() == 0)) {
            throw new IllegalArgumentException(
                "The value can not be null or empty.");
        }

        this.value = value;
        this.description = description;
    }

    /**
     * Returns the description.
     *
     * @return the description or null
     */
    public String getDescription() {
        return description;
    }

    /**
     * Returns the value.
     *
     * @return the value
     */
    public String getValue() {
        return value;
    }

    /**
     * Sets the description of this entry.
     *
     * @param description the description
     */
    public void setDescription(String description) {
        this.description = description;
    }

    /**
     * Sets the value of this entry.<br>
     * This method has restricted accessibility in order to minimize the risk
     * of  unintentionally creating two or more CVEntries in a
     * ControlledVocabulary, by some client-side class.  To modify the value
     * of a CVEntry that is in a ControlledVocabulary  ask the CV to set the
     * value of the entry in a checked way. It is always possible to create a
     * new CVEntry with the new value and copy  other data (description) from
     * the old one, remove rhe old entry from the CV  and add the new antry to
     * the CV.
     *
     * @param value the new value of the entry
     *
     * @throws IllegalArgumentException if value == null
     *
     * @see ControlledVocabulary#modifyEntryValue(CVEntry, String)
     */
    void setValue(String value) {
        if ((value == null) || (value.length() == 0)) {
            throw new IllegalArgumentException(
                "The value can not be null or empty.");
        }

        this.value = value;
    }

    /**
     * Overrides <code>Object</code>'s toString() method to just return  the
     * value of this entry.<br>
     * This way this object can easily be used directly in Lists, ComboBoxes
     * etc.
     *
     * @return the value
     */
    public String toString() {
        return value;
    }

    /**
     * Overrides <code>Object</code>'s equals method by checking all  fields of
     * the other object to be equal to all fields in this  object.
     *
     * @param obj the reference object with which to compare
     *
     * @return true if this object is the same as the obj argument; false
     *         otherwise
     */
    public boolean equals(Object obj) {
        if (obj == null) {
            // null is never equal
            return false;
        }

        if (obj == this) {
            // same object reference 
            return true;
        }

        if (!(obj instanceof CVEntry)) {
            // it should be a MediaDescriptor object
            return false;
        }

        // check the fields
        CVEntry other = (CVEntry) obj;

        if (value.equals(other.getValue())) {
            if ((description == null) && (other.getDescription() == null)) {
                return true;
            } else {
                return description.equals(other.getDescription());
            }
        }

        return false;
    }
}
