/*
 * File:     ToolID.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.tool;

import java.io.Serializable;

import java.util.Vector;


/**
 * A class whose objects identify Tools. Eudico Tools are principally
 * identified by their category and their name, and their class name. A ToolID
 * is also able to inform the user if the corresponding Tool is available as
 * server based tool and/or client based tool. The user's choice is
 * administered by the ToolID, which is in turn used by the ToolDatabase to
 * create an instance of the specified Tool. In the long run server based
 * Tools will also have client versions since we may want them to work on
 * locally available data sets. ToolIDs can be associated with ToolConditions.
 * For the corresponding Tool to be applicable all of these ToolConditions
 * have to be met.
 *
 * @author Hennie Brugman
 * @author Albert Russel
 * @version 29-Apr-1999
 */
public class ToolID implements Serializable, Comparable {
    /** Constants to indicate where a Tool is to be instantiated. */
    public static short LOCAL = 1;

    /** Holds value of property DOCUMENT ME! */
    public static short SERVER = 2;

    /** The category to which the Tool belongs, for example 'Viewers' */
    private String category;

    /**
     * The name of the Tool as it is shown in the Tool Tree. For example 'Time
     * Line Viewer'.
     */
    private String name;

    /**
     * The name of the .class file that implements this tool.  For example
     * 'mpi.eudico.tool.TimeLineView'.
     */
    private String className;

    /**
     * If true, there is a version of the corresponding Tool that can run on
     * clients. Example: Viewers.
     */
    private boolean clientVersionAvailable;

    /**
     * If true, there is a version of the corresponding Tool that can run on
     * the server. Example: a search tool.
     */
    private boolean serverVersionAvailable;

    /**
     * The Tool specified by this ToolID is to be created either LOCALly or on
     * the SERVER.
     */
    private short requestedHost;

    /**
     * A list of conditions that have to be met by ToolAdministrators for the
     * associated Tool to be applicable.
     */
    private Vector toolConditions;

    /**
     * If false, this ToolID is intended for demonstration purposes. It's
     * information will show up in the Eudico browser, but dimmed and
     * disabled.
     */
    private boolean isImplemented;

    /** Shows the status of the Tool */
    private boolean isEnabled = true;

    /**
     * Creates a ToolID
     *
     * @param theCategory The Tool category.
     * @param theName The Tool name.
     * @param theClassName The implementing .class file.
     * @param clientVersion DOCUMENT ME!
     * @param serverVersion DOCUMENT ME!
     * @param implemented DOCUMENT ME!
     */
    public ToolID(String theCategory, String theName, String theClassName,
        boolean clientVersion, boolean serverVersion, boolean implemented) {
        category = theCategory;
        name = theName;
        className = theClassName;
        clientVersionAvailable = clientVersion;
        serverVersionAvailable = serverVersion;

        isImplemented = implemented;

        requestedHost = LOCAL;

        toolConditions = new Vector();
    }

    /**
     * DOCUMENT ME!
     *
     * @return the ToolID category.
     */
    public String getCategory() {
        return category;
    }

    /**
     * DOCUMENT ME!
     *
     * @return the ToolID name.
     */
    public String getName() {
        return name;
    }

    /**
     * DOCUMENT ME!
     *
     * @return the ToolID className.
     */
    public String getClassName() {
        return className;
    }

    /**
     * Returns true if the corresponding Tool is available to run on  clients.
     *
     * @return DOCUMENT ME!
     */
    public boolean hasClientVersion() {
        return clientVersionAvailable;
    }

    /**
     * Returns true if the corresponding Tool is available to run on the
     * server.
     *
     * @return DOCUMENT ME!
     */
    public boolean hasServerVersion() {
        return serverVersionAvailable;
    }

    /**
     * Specify user's request if a Tool is to be created and run on the client
     * or on the server.
     *
     * @param localOrServer DOCUMENT ME!
     */
    public void setHost(short localOrServer) {
        requestedHost = localOrServer;
    }

    /**
     * Compares two ToolID's according to the lexicografical ordering of their
     * category and name Strings.
     *
     * @param toolID The other ToolID with which this one is to be compared.
     *
     * @return the value 0 if both ToolID's are equal, a value less than 0 if
     *         this  ToolID is lexicographically less than the toolID
     *         parameter, and a value  greater than 0 if this ToolID is
     *         lexicographically greater than the toolID  parameter.
     */
    public int compareTo(Object toolID) {
        String s1 = category + name;
        String s2 = ((ToolID) toolID).getCategory() +
            ((ToolID) toolID).getName();

        return s1.compareTo(s2);
    }

    /**
     * Returns conditions that have to be met for this Tool to be applicable.
     *
     * @return DOCUMENT ME!
     */
    public Vector getConditions() {
        return toolConditions;
    }

    /**
     * DOCUMENT ME!
     *
     * @param theToolCondition DOCUMENT ME!
     */
    public void addCondition(ToolCondition theToolCondition) {
        toolConditions.add(theToolCondition);
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean isImplemented() {
        return isImplemented;
    }

    /**
     * DOCUMENT ME!
     *
     * @param b DOCUMENT ME!
     */
    public void setEnabled(boolean b) {
        isEnabled = b;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public boolean isEnabled() {
        return isEnabled;
    }
}
