/*
 * File:     TimeRelation.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.eudico.util;

import mpi.eudico.server.corpora.clom.AnnotationCore;


/**
 * Possible time relations between an AnnotationCore(-interval) and an externally given interval
 * @author Alexander Klassmann
 * @version July 2004
 */
public class TimeRelation {
    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean overlaps(AnnotationCore annotationCore,
        long intervalBegin, long intervalEnd) {
        return !doesNotOverlap(annotationCore, intervalBegin, intervalEnd);
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean doesNotOverlap(AnnotationCore annotationCore,
        long intervalBegin, long intervalEnd) {
        return (annotationCore.getEndTimeBoundary() <= intervalBegin) ||
        (annotationCore.getBeginTimeBoundary() >= intervalEnd);
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean isInside(AnnotationCore annotationCore,
        long intervalBegin, long intervalEnd) {
        return (annotationCore.getBeginTimeBoundary() >= intervalBegin) &&
        (annotationCore.getEndTimeBoundary() <= intervalEnd);
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean isNotInside(AnnotationCore annotationCore,
        long intervalBegin, long intervalEnd) {
        return !isInside(annotationCore, intervalBegin, intervalEnd);
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean overlapsOnLeftSide(
        AnnotationCore annotationCore, long intervalBegin, long intervalEnd) {
        return (annotationCore.getBeginTimeBoundary() < intervalBegin) &&
        (intervalBegin < annotationCore.getEndTimeBoundary()) &&
        (annotationCore.getEndTimeBoundary() < intervalEnd);
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @return boolean
     */
    final public static boolean overlapsOnRightSide(
        AnnotationCore annotationCore, long intervalBegin, long intervalEnd) {
        return (intervalBegin < annotationCore.getBeginTimeBoundary()) &&
        (annotationCore.getBeginTimeBoundary() < intervalEnd) &&
        (intervalEnd < annotationCore.getEndTimeBoundary());
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalBegin
     * @param distance
     * @return boolean
     */
    final public static boolean isWithinLeftDistance(
        AnnotationCore annotationCore, long intervalBegin, long distance) {
        return (distance == Long.MAX_VALUE) ||
        ((annotationCore.getBeginTimeBoundary() > (intervalBegin - distance)) &&
        (annotationCore.getEndTimeBoundary() <= (intervalBegin + distance)));
    }

    /**
     *
     * @param AnnotationCore
     * @param intervalEnd
     * @param distance
     * @return boolean
     */
    final public static boolean isWithinRightDistance(
        AnnotationCore annotationCore, long intervalEnd, long distance) {
        return (distance == Long.MAX_VALUE) ||
        ((annotationCore.getBeginTimeBoundary() > (intervalEnd - distance)) &&
        (annotationCore.getEndTimeBoundary() <= (intervalEnd + distance)));
    }

    /**
     * Returns negative of isWithinLeftDistance()
     * @param AnnotationCore
     * @param intervalBegin
     * @param leftDistance
     * @return boolean
     */
    final public static boolean isBeforeLeftDistance(
        AnnotationCore annotationCore, long intervalBegin, long distance) {
        return annotationCore.getEndTimeBoundary() < (intervalBegin - distance);
    }

    /**
     * Returns negative of isWithinRightDistance()
     * @param AnnotationCore
     * @param intervalEnd
     * @param rightDistance
     * @return boolean
     */
    final public static boolean isAfterRightDistance(
        AnnotationCore annotationCore, long intervalEnd, long distance) {
        return annotationCore.getBeginTimeBoundary() > (intervalEnd + distance);
    }

    /**
     * Returns true if intervalBegin - distance &lt; annotationBegin AND
     * annotationEnd &lt; intervalEnd + distance
     * @param AnnotationCore
     * @param intervalBegin
     * @param intervalEnd
     * @param distance
     * @return boolean
     */
    final public static boolean isWithinDistance(
        AnnotationCore annotationCore, long intervalBegin, long intervalEnd,
        long distance) {
        return (distance == Long.MAX_VALUE) ||
        ((annotationCore.getBeginTimeBoundary() > (intervalBegin - distance)) &&
        (annotationCore.getEndTimeBoundary() <= (intervalEnd + distance)));
    }
}
