/*
 * File:     DependentConstraint.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.model;

import java.util.HashMap;


/**
 * Created on Aug 18, 2004
 * @author Alexander Klassmann
 * @version Aug 18, 2004
 */
public class DependentConstraint extends AbstractConstraint {
    /** Holds TEMPORAL or STRUCTURAL */
    private final String mode;

    /** Holds ANY or NONE */
    private final String quantifier;

    /** Holds number of referential constraint */
    private final int refConstraint;

    /**
     * DOCUMENT ME!
     *
     * @param index constraint number within a query
     * @param fieldType fieldType resp. Track
     * @param patternString string/regular expression to be searched
     * @param lowerBoundary negative number (of units) (e.g. 0, -1, -2, ... -X)
     * @param upperBoundary positive number (of units) (e.g. 0, 1, 2 ... +X)
     * @param unit search unit in which should be searched (in respect to
     *        referential constraint)
     * @param refConstraint referential constraint
     * @param isRegEx string or regular expression ?
     * @param isCaseSensitive case sensitive string search ?
     * @param attributes should contain (as strings) attribute names (key) and
     *        values (value)
     */
    public DependentConstraint(int index, int refConstraint, String tierName,
        String patternString, long lowerBoundary, long upperBoundary,
        String unit, boolean isRegEx, boolean isCaseSensitive,
        HashMap attributes) {
        this(index, refConstraint, AnchorConstraint.ANY, tierName,
            patternString, Constraint.STRUCTURAL, lowerBoundary, upperBoundary,
            unit, isRegEx, isCaseSensitive, attributes);
    }

    /**
     * DOCUMENT ME!
     *
     * @param index constraint number within a query
     * @param type CGN or EAF
     * @param fieldType fieldType resp. Track
     * @param patternString string/regular expression to be searched
     * @param lowerBoundary negative number (of units) (e.g. 0, -1, -2, ... -X)
     * @param upperBoundary positive number (of units) (e.g. 0, 1, 2 ... +X)
     * @param unit search unit in which should be searched (in respect to
     *        referential constraint)
     * @param refConstraint referential constraint
     * @param isRegEx string or regular expression ?
     * @param isCaseSensitive case sensitive string search ?
     * @param attributes should contain (as strings) attribute names (key) and
     *        values (value)
     */
    public DependentConstraint(int index, int refConstraint, String quantifier,
        String tierName, String patternString, String mode, long lowerBoundary,
        long upperBoundary, String unit, boolean isRegEx,
        boolean isCaseSensitive, HashMap attributes) {
        super(index, new String[] { tierName }, patternString, lowerBoundary,
            upperBoundary, unit, isRegEx, isCaseSensitive, attributes);
        this.mode = mode;
        this.quantifier = quantifier;
        this.refConstraint = refConstraint;
    }

    public String getTierName() {
        return tierNames[0];
    }

    public String getMode() {
        return mode;
    }

    public String getQuantifier() {
        return quantifier;
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public int getRefConstraintNr() {
        return refConstraint;
    }

    public boolean equals(Object object) {
        if (!(object instanceof DependentConstraint)) {
            return false;
        }

        DependentConstraint constraint = (DependentConstraint) object;

        if (!constraint.getMode().equals(getMode())) {
            return false;
        }

        if (!constraint.getTierName().equals(getTierName())) {
            return false;
        }

        if (constraint.getRefConstraintNr() != getRefConstraintNr()) {
            return false;
        }

        return super.equals(object);
    }
}
