/*
 * File:     AnchorConstraintPanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.viewer;

import mpi.search.SearchLocale;

import mpi.search.content.model.CorpusType;

import mpi.search.content.query.model.AnchorConstraint;

import java.awt.Font;
import java.awt.Graphics;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.SwingConstants;
import javax.swing.border.EmptyBorder;


/**
 * Created on Aug 18, 2004
 * @author Alexander Klassmann
 * @version Aug 18, 2004
 */
public class AnchorConstraintPanel extends AbstractConstraintPanel {
    final public String ALL_TIERS = "Search.Constraint.AllTiers";
    final private JLabel lockedConstraintLabel = new JLabel();
    private String lockedTierName = null;
    private String lockedPattern = null;

    public AnchorConstraintPanel(QueryPanel parentPanel, Font font,
        CorpusType type) {
        super(parentPanel, font, 1, type);

        // 'head line' of constraint
        titleComponent = new JLabel(SearchLocale.getString("Search.Query.Find")
                                                .toUpperCase(),
                SwingConstants.CENTER);
        titleComponent.setBorder(new EmptyBorder(10, 0, 5, 0));

        tierComboBox = new JComboBox(type.getTierNames()) {
                    public void paint(Graphics g) {
                        super.paint(g);
                        tierComboBoxWidth = this.getPreferredSize().width;
                    }
                };

        if (type.allowsSearchOverMultipleTiers()) {
            tierComboBox.insertItemAt(ALL_TIERS, 0);
        }

        lockedConstraintLabel.setHorizontalAlignment(JLabel.CENTER);
        framedPanel.add(lockedConstraintLabel, "locked");
        makeLayout();
        tierComboBox.setSelectedIndex(0);
    }

    public AnchorConstraint getConstraint() {
        return new AnchorConstraint(constraintNr, getTierNames(), getPattern(),
            relationPanel.getLowerBoundary(), relationPanel.getUpperBoundary(),
            relationPanel.getUnit(), isRegEx(), isCaseSensitive(),
            getAttributes());
    }

    protected void setTierName(String tierName) {
        for (int i = 0; i < type.getIndexTierNames().length; i++) {
            if (type.getIndexTierNames()[i].equals(tierName.toUpperCase())) {
                lockedTierName = tierName.toUpperCase();
                framedPanelLayout.show(framedPanel, "locked");

                return;
            }
        }

        //if lockedField was set, reset.
        if (lockedTierName != null) {
            lockedTierName = null;
            lockedPattern = null;
            framedPanelLayout.show(framedPanel, "");
        }

        super.setTierName(tierName);
    }

    protected void setTierNames(String[] tierNames) {
        if (tierNames.length == 1) {
            setTierName(tierNames[0]);
        } else {
            tierComboBox.setSelectedItem(ALL_TIERS);
        }
    }

    public String getTierName() {
        return (lockedTierName == null) ? super.getTierName() : lockedTierName;
    }

    protected String[] getTierNames() {
        String tierName = getTierName();

        return tierName.equals(ALL_TIERS) ? type.getTierNames()
                                          : new String[] { tierName };
    }

    protected void setPattern(String pattern) {
        if (lockedTierName != null) {
            lockedPattern = pattern;
            lockedConstraintLabel.setText(type.getUnabbreviatedTierName(
                    lockedTierName) + " " + lockedPattern);
        } else {
            super.setPattern(pattern);
        }
    }

    protected String getPattern() {
        return (lockedPattern == null) ? super.getPattern() : lockedPattern;
    }

    public void setConstraint(AnchorConstraint c) {
        setTierNames(c.getTierNames());
        super.setConstraint(c);
    }
}
