/*
 * File:     DependentConstraintPanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.viewer;

import mpi.search.SearchLocale;

import mpi.search.content.model.CorpusType;

import mpi.search.content.query.model.Constraint;
import mpi.search.content.query.model.DependentConstraint;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;


/**
 * Created on Aug 18, 2004
 * @author Alexander Klassmann
 * @version Aug 18, 2004
 */
public class DependentConstraintPanel extends AbstractConstraintPanel {
    final protected JComboBox modeComboBox = new JComboBox(Constraint.MODES);

    public DependentConstraintPanel(QueryPanel parentPanel, Font font,
        int constraintNr, CorpusType type) {
        super(parentPanel, font, constraintNr, type);

        titleComponent = new JPanel(new FlowLayout());
        titleComponent.add(new JLabel(SearchLocale.getString(
                    "Search.Query.With").toUpperCase()));

        if (type.allowsTemporalConstraints()) {
            titleComponent.add(modeComboBox);
        }

        titleComponent.add(new JLabel(SearchLocale.getString(
                    "Search.Query.Constraint").toUpperCase()));
        modeComboBox.setRenderer(new LocalizeListCellRenderer());
        modeComboBox.addItemListener(new ItemListener() {
                public void itemStateChanged(ItemEvent e) {
                    if (e.getStateChange() == ItemEvent.SELECTED) {
                        updateMode();
                    }
                }
            });
        tierComboBox = new JComboBox() {
                    public Dimension getPreferredSize() {
                        return new Dimension(tierComboBoxWidth,
                            super.getPreferredSize().height);
                    }
                };

        makeLayout();

        //setMode(Constraint.STRUCTURAL);
        updateMode();
    }

    protected void update() {
        if (queryPanel.getTierNameOfConstraintNr(
                    relationPanel.getRefConstraintNr()) == null) {
            modeComboBox.setSelectedItem(Constraint.TEMPORAL);
            modeComboBox.setEnabled(false);
        } else {
            modeComboBox.setEnabled(true);
        }

        updateTierComboBox();
    }

    protected void updateTierComboBox() {
        if (Constraint.TEMPORAL.equals(getMode())) {
            updateComboBox(tierComboBox, type.getTierNames());
        } else {
            String referentialTierName = queryPanel.getTierNameOfConstraintNr(relationPanel.getRefConstraintNr());
            updateComboBox(tierComboBox,
                type.getRelatedTiers(referentialTierName));
            updateUnitComboBox();
        }
    }

    /**
     * Fills the unitComboBox with all SearchUnits which share the tier on this
     * constraint and the tier of the constraint it refers to
     */
    protected void updateUnitComboBox() {
        if (Constraint.STRUCTURAL.equals(getMode())) {
            String referentialTierName = queryPanel.getTierNameOfConstraintNr(getRefConstraintNr());
            String[] possibleUnits = (type.getPossibleUnitsFor((String) tierComboBox.getSelectedItem(),
                    referentialTierName));
            updateComboBox(relationPanel.getUnitComboBox(), possibleUnits);
        }
    }

    protected String getMode() {
        return (String) modeComboBox.getSelectedItem();
    }

    protected void setMode(String mode) {
        modeComboBox.setSelectedItem(mode);
    }

    protected void updateMode() {
        update();
        relationPanel.setDistanceMode((String) modeComboBox.getSelectedItem());
    }

    protected int getRefConstraintNr() {
        return relationPanel.getRefConstraintNr();
    }

    public DependentConstraint getConstraint() {
        return new DependentConstraint(constraintNr, getRefConstraintNr(),
            patternPanel.getQuantifier(), getTierName(), getPattern(),
            getMode(), relationPanel.getLowerBoundary(),
            relationPanel.getUpperBoundary(), relationPanel.getUnit(),
            isRegEx(), isCaseSensitive(), getAttributes());
    }

    /**
     * DOCUMENT ME!
     *
     * @param c DOCUMENT ME!
     */
    public void setConstraint(DependentConstraint c) {
        setMode(c.getMode());
        setTierName(c.getTierName());
        relationPanel.setRefConstraintNr(c.getRefConstraintNr());
        super.setConstraint(c);
    }

    //implementation of ItemListener
    public void itemStateChanged(ItemEvent e) {
        if (e.getStateChange() == ItemEvent.SELECTED) {
            if (e.getSource() == tierComboBox) {
                updateUnitComboBox();
            }
        }

        super.itemStateChanged(e);
    }
}
