/*
 * File:     PatternPanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.viewer;

import mpi.search.SearchLocale;

import mpi.search.content.model.CorpusType;

import mpi.search.content.query.model.AnchorConstraint;

import mpi.util.gui.SplittedExplicativeJComboBoxPanel;

import java.awt.CardLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import java.util.List;

import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;


/**
 * Created on Jul 13, 2004
 * @author Alexander Klassmann
 * @version Jul 13, 2004
 */
public class PatternPanel extends JPanel implements ItemListener {
    final private static int MAX_ENTRY_SPLITTED_COMBOBOX = 51;

    /** Holds value of property DOCUMENT ME! */
    final protected JTextField textField = new JTextField(14);

    /** Holds cardLayout */
    final protected CardLayout inputLayout = new CardLayout();

    /** Holds components to type in resp. choose a pattern */
    final protected JPanel inputPanel = new JPanel(inputLayout);

    /** Holds value of property DOCUMENT ME */
    final protected JComboBox quantifierComboBox = new JComboBox(AnchorConstraint.QUANTIFIERS);
    final private CorpusType type;

    public PatternPanel(CorpusType type, JComboBox tierComboBox,
        int constraintNr, Font font) {
        this.type = type;

        if (font != null) {
            textField.setFont(font.deriveFont(Font.PLAIN, 16f));
        } else {
            textField.setFont(textField.getFont().deriveFont(Font.PLAIN, 16f));
        }

        inputPanel.add(textField, "");

        for (int j = 0; j < type.getTierNames().length; j++) {
            String tierName = type.getTierNames()[j];
            List closedVoc = type.getClosedVoc(tierName);

            if (closedVoc != null) {
                SplittedExplicativeJComboBoxPanel localPanel = new SplittedExplicativeJComboBoxPanel(closedVoc,
                        '(', MAX_ENTRY_SPLITTED_COMBOBOX) {
                        public Dimension getPreferredSize() {
                            return new Dimension(textField.getPreferredSize().width,
                                super.getPreferredSize().height);
                        }
                    };

                localPanel.setEditable(type.isClosedVocEditable(closedVoc));
                inputPanel.add(localPanel, tierName);
            }
        }

        setLayout(new GridBagLayout());

        GridBagConstraints c = new GridBagConstraints();

        if ((constraintNr == 0) || !type.allowsQuantifierNO()) {
            add(new JLabel(SearchLocale.getString(AnchorConstraint.ANY)));
        } else {
            add(quantifierComboBox);
        }

        add(new JLabel(" " +
                SearchLocale.getString("Search.Constraint.AnnotationOnTier") +
                " "), c);
        add(tierComboBox, c);

        if ((constraintNr == 1) || !type.allowsQuantifierNO()) {
            add(new JLabel(" " +
                    SearchLocale.getString("Search.Constraint.That")), c);
        }

        add(new JLabel(" " +
                SearchLocale.getString("Search.Constraint.Matches") + " "), c);
        add(inputPanel, c);
        inputLayout.show(inputPanel, "");
        quantifierComboBox.setRenderer(new LocalizeListCellRenderer());
        quantifierComboBox.setSelectedItem(AnchorConstraint.ANY);
        tierComboBox.addItemListener(this);
        tierComboBox.setRenderer(new TierListCellRenderer(type));
    }

    public void itemStateChanged(ItemEvent e) {
        if (e.getStateChange() == ItemEvent.SELECTED) {
            if (type.isClosedVoc((String) e.getItem())) {
                inputLayout.show(inputPanel, (String) e.getItem());
            } else {
                inputLayout.show(inputPanel, "");
            }

            inputPanel.setLocale(type.getDefaultLocale((String) e.getItem()));
            validate();
            repaint();
        }
    }

    public Component getVisibleInputComponent() {
        Component[] comps = inputPanel.getComponents();

        for (int i = 0; i < comps.length; i++) {
            if (comps[i].isVisible()) {
                return comps[i];
            }
        }

        return null;
    }

    public Component getDefaultInputComponent() {
        return textField;
    }

    public String getPattern() {
        Component c = getVisibleInputComponent();

        return (c instanceof JTextField) ? ((JTextField) c).getText()
                                         : ((SplittedExplicativeJComboBoxPanel) c).getSelectedItem()
                                            .toString();
    }

    /**
     * DOCUMENT ME!
     *
     * @param regEx DOCUMENT ME!
     */
    public void setPattern(String pattern) {
        Component c = getVisibleInputComponent();

        if (c instanceof JTextField) {
            ((JTextField) c).setText(pattern);
        } else {
            ((SplittedExplicativeJComboBoxPanel) c).setSelectedItem(pattern);
        }
    }

    public String getQuantifier() {
        return (String) quantifierComboBox.getSelectedItem();
    }

    public void setQuantifier(String quantifier) {
        quantifierComboBox.setSelectedItem(quantifier);
    }
}
