/*
 * File:     QueryPanel.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.viewer;

import mpi.search.content.model.CorpusType;

import mpi.search.content.query.model.AnchorConstraint;
import mpi.search.content.query.model.ContentQuery;
import mpi.search.content.query.model.RestrictedAnchorConstraint;

import mpi.search.content.result.model.ContentResult;

import mpi.search.viewer.AbstractQueryPanel;

import javax.swing.JPanel;


/**
 * The SearchConfigPanel is the GUI for defining a complete Query. It is
 * composed of one or more Constraint-Panels
 *
 * @author Alexander Klassmann
 * @version July 2004
 */
public class QueryPanel extends AbstractQueryPanel {
    /** Holds value of property DOCUMENT ME! */
    private final CorpusType type;

    /**
     * Creates a new QueryPanel instance
     *
     * @param theSearchableDataSet
     *            DOCUMENT ME!
     * @param max_query
     *            DOCUMENT ME!
     */
    public QueryPanel(CorpusType type) {
        this.type = type;

        addConstraint();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public ContentQuery getQuery() {
        ContentQuery query = new ContentQuery(type);

        AnchorConstraintPanel aPanel = (AnchorConstraintPanel) constraintGridPanel.getComponent(0);
        query.setAnchorConstraint(aPanel.getConstraint());

        DependentConstraintPanel dPanel;

        for (int i = 1; i < constraintGridPanel.getComponentCount(); i++) {
            dPanel = (DependentConstraintPanel) constraintGridPanel.getComponent(i);
            query.addDependentConstraint(dPanel.getConstraint());
        }

        return query;
    }

    /**
     * Configure the panel with a previously defined Query.
     *
     * @param theQuery
     *            DOCUMENT ME!
     */
    public void setQuery(ContentQuery query) {
        constraintGridPanel.removeAll();

        if (query != null) {
            AnchorConstraint ac = query.getAnchorConstraint();

            AnchorConstraintPanel panel = (ac instanceof RestrictedAnchorConstraint)
                ? new RestrictedAnchorConstraintPanel(this,
                    getParent().getFont(), type, (RestrictedAnchorConstraint) ac)
                : createAnchorConstraintPanel();

            constraintGridPanel.add(panel);
            panel.setConstraint(ac);

            for (int i = 1; i < query.size(); i++) {
                DependentConstraintPanel dPanel = createDependentConstraintPanel();
                constraintGridPanel.add(dPanel); // must be added before

                // constraint is set
                dPanel.setConstraint(query.getDependentConstraint(i));
            }

            if (getTopLevelAncestor() != null) {
                getTopLevelAncestor().validate();
            }

            updateActions();
        }
    }

    public void newRestrictedQuery(ContentResult result, String comment) {
        if (result.getMatchCount() == 0) {
            return;
        }

        constraintGridPanel.removeAll();

        RestrictedAnchorConstraint ac = new RestrictedAnchorConstraint(result,
                comment);
        constraintGridPanel.add(new RestrictedAnchorConstraintPanel(this,
                getParent().getFont(), type, ac));
        constraintGridPanel.add(createDependentConstraintPanel());
        getTopLevelAncestor().validate();
        updateActions();
    }

    protected JPanel createConstraintPanel() {
        return (constraintGridPanel.getComponentCount() == 0)
        ? (AbstractConstraintPanel) createAnchorConstraintPanel()
        : createDependentConstraintPanel();
    }

    private AnchorConstraintPanel createAnchorConstraintPanel() {
        return new AnchorConstraintPanel(this, getFont(), type);
    }

    private DependentConstraintPanel createDependentConstraintPanel() {
        return new DependentConstraintPanel(this, getFont(),
            constraintGridPanel.getComponentCount() + 1, type);
    }

    /**
     * Returns the selected Tier in the Constraint Nr i (resp. i+1) if multiple
     * tiers are selected, then it returns null
     *
     * @param i
     *            DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected String getTierNameOfConstraintNr(int i) {
        AbstractConstraintPanel constraintPanel = (AbstractConstraintPanel) constraintGridPanel.getComponent(i -
                1);
        String tierName = constraintPanel.getTierName();

        if (constraintPanel instanceof AnchorConstraintPanel) {
            if (((AnchorConstraintPanel) constraintPanel).getTierNames().length > 1) {
                tierName = null;
            }
        }

        return tierName;
    }

    /**
     * If in one Constraint the selected tier has changed, all following
     * constraints, which refer to this constraint are notified
     *
     * @param constraintPanel
     *            DOCUMENT ME!
     */
    protected void notifyOtherConstraints(int index) {
        // component may not (yet) be visible -> do nothing
        if (getComponentCount() < index) {
            return;
        }

        for (int i = index; i < constraintGridPanel.getComponentCount(); i++) {
            DependentConstraintPanel dPanel = (DependentConstraintPanel) constraintGridPanel.getComponent(i);

            if (dPanel.getRefConstraintNr() == index) {
                dPanel.update();
            }
        }
    }

    protected void updateActions() {
        if (constraintGridPanel.getComponent(0) instanceof RestrictedAnchorConstraintPanel) {
            deleteConstraintAction.setEnabled(constraintGridPanel.getComponentCount() > 2);
            addConstraintAction.setEnabled(constraintGridPanel.getComponentCount() < max_constraint);
        } else {
            super.updateActions();
        }
    }
}
