/*
 * File:     Query2Xml.java
 * Project:  MPI Linguistic Application
 * Date:     03 April 2006
 *
 * Copyright (C) 2001-2006  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package mpi.search.content.query.xml;

import mpi.search.content.query.model.*;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.util.Date;


/**
 * Created on Sep 29, 2004
 *
 * @author Alexander Klassmann
 * @version November 2004
 */
public class Query2Xml {
    public static void translate(String fileName, ContentQuery query)
        throws IOException {
        File outputFile = new File(fileName);
        OutputStreamWriter out = new OutputStreamWriter(new FileOutputStream(
                    outputFile));
        out.write("<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>\n");
        writeQueryToStream(query, out);
        out.close();
    }

    /**
     * @param query
     * @param out
     * @throws IOException
     */
    public static void writeQueryToStream(ContentQuery query,
        OutputStreamWriter out) throws IOException {
        out.write("<query date=\"" +
            new Date(System.currentTimeMillis()).toString() + "\">\n");

        out.write("<description>");
        out.write(query.toString());
        out.write("</description>\n");

        AnchorConstraint anchorConstraint = query.getAnchorConstraint();

        if (anchorConstraint != null) {
            out.write("<anchorConstraint id=\"");
            out.write("" + anchorConstraint.getNr());
            writeAttributes(out, anchorConstraint);
            out.write("\">\n");

            String[] tierNames = anchorConstraint.getTierNames();

            for (int j = 0; j < tierNames.length; j++) {
                out.write("<tier>" + tierNames[j] + "</tier>");
            }

            out.write("<pattern>" + anchorConstraint.getPattern() +
                "</pattern>\n");
            out.write("</anchorConstraint>");
        }

        for (int i = 1; i < query.size(); i++) {
            DependentConstraint dependentConstraint = query.getDependentConstraint(i);
            out.write("<dependentConstraint id=\"");
            out.write("" + dependentConstraint.getNr());
            out.write("\" mode=\"");
            out.write(dependentConstraint.getMode());
            out.write("\" quantifier=\"");
            out.write(dependentConstraint.getQuantifier());
            writeAttributes(out, dependentConstraint);
            out.write("\" id_ref=\"");
            out.write("" + dependentConstraint.getRefConstraintNr());
            out.write("\">\n");
            out.write("<tier>" + dependentConstraint.getTierName() + "</tier>");
            out.write("<pattern>" + dependentConstraint.getPattern() +
                "</pattern>\n");
            out.write("</dependentConstraint>\n");
        }

        out.write("</query>\n");
    }

    public static void writeAttributes(OutputStreamWriter out,
        Constraint constraint) throws IOException {
        out.write("\" regularExpression=\"");
        out.write("" + constraint.isRegEx());
        out.write("\" caseSensitive=\"");
        out.write("" + constraint.isCaseSensitive());

        if (constraint.getLowerBoundary() != Long.MIN_VALUE) {
            out.write("\" from=\"");
            out.write("" + constraint.getLowerBoundary());
        }

        if (constraint.getUpperBoundary() != Long.MAX_VALUE) {
            out.write("\" to=\"");
            out.write("" + constraint.getUpperBoundary());
        }

        out.write("\" unit=\"");
        out.write(constraint.getUnit());
    }
}
