/*
 * File:     ElanFrame2.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator;

import mpi.eudico.client.annotator.commands.BackupCA;
import mpi.eudico.client.annotator.commands.Command;
import mpi.eudico.client.annotator.commands.CommandAction;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;
import mpi.eudico.client.annotator.commands.StoreCommand;
import mpi.eudico.client.annotator.commands.global.*;

import mpi.eudico.client.annotator.grid.GridViewer;

import mpi.eudico.client.annotator.gui.ElanMenuItem;

import mpi.eudico.client.annotator.linkedmedia.LinkedFileDescriptorUtil;
import mpi.eudico.client.annotator.linkedmedia.MediaDescriptorUtil;

import mpi.eudico.client.annotator.player.JMFGraphicMediaPlayer;

import mpi.eudico.client.annotator.svg.SVGPrefs;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileUtility;
import mpi.eudico.client.annotator.util.FrameConstants;

import mpi.eudico.client.annotator.viewer.InterlinearViewer;
import mpi.eudico.client.annotator.viewer.SubtitleViewer;
import mpi.eudico.client.annotator.viewer.TextViewer;
import mpi.eudico.client.annotator.viewer.TimeLineViewer;
import mpi.eudico.client.annotator.viewer.TimeSeriesViewer;

import mpi.eudico.client.mac.MacApplicationListener;

import mpi.eudico.p2p.*;

import mpi.eudico.server.corpora.clom.Transcription;
import mpi.eudico.server.corpora.clom.TranscriptionStore;

import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.ACMTranscriptionStore;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.io.File;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Vector;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.ButtonGroup;
import javax.swing.ComponentInputMap;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.JTextComponent;
import javax.swing.text.Keymap;


/**
 * DOCUMENT ME! $Id: ElanFrame2.java 11167 2007-12-10 08:53:41Z hasloe $
 *
 * @author $Author$
 * @version $Revision$
 */
public class ElanFrame2 extends JFrame implements ActionListener,
    ElanLocaleListener, MacApplicationListener, FrameConstants, PreferencesUser {
    // load some keybindings for the Mac
    static {
        if (System.getProperty("os.name").indexOf("Mac") > -1) {
            JTextComponent.KeyBinding[] bind = new JTextComponent.KeyBinding[] {
                    new JTextComponent.KeyBinding(KeyStroke.getKeyStroke(
                            KeyEvent.VK_C,
                            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()),
                        DefaultEditorKit.copyAction),
                    new JTextComponent.KeyBinding(KeyStroke.getKeyStroke(
                            KeyEvent.VK_X,
                            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()),
                        DefaultEditorKit.cutAction),
                    new JTextComponent.KeyBinding(KeyStroke.getKeyStroke(
                            KeyEvent.VK_V,
                            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()),
                        DefaultEditorKit.pasteAction),
                    new JTextComponent.KeyBinding(KeyStroke.getKeyStroke(
                            KeyEvent.VK_A,
                            Toolkit.getDefaultToolkit().getMenuShortcutKeyMask()),
                        DefaultEditorKit.selectAllAction)
                };

            JTextComponent comp = new JTextField();
            Keymap map = JTextComponent.getKeymap(JTextComponent.DEFAULT_KEYMAP);
            JTextComponent.loadKeymap(map, bind, comp.getActions());
        }
    }

    /** a list of actions for menu containers and a few transcription independent
     * actions */
    private ArrayList menuActions = new ArrayList(20);

    /** a map for actions that are added to the menubar's action map
     * this storage is needed for detaching and re-attaching actions to the menu bar */
    private HashMap registeredActions = new HashMap();
    private Transcription transcriptionForThisFrame;
    private JMenuBar menuBar;
    private JMenu menuFile;
    private JMenuItem menuItemFileNew;
    private JMenuItem menuItemFileOpen;
    private JMenu menuRecentFiles;
    private JMenuItem menuItemFileExit;
    private JMenu menuBackup;
    private JMenu menuImport;
    private JMenu menuExport;
    private JMenuItem menuItemShoeboxImport;
    private JMenuItem menuItemCHATImport;
    private JMenuItem menuItemTranscriberImport;
    private JMenuItem menuItemDiscoverDoc;
    private JMenuItem menuItemPublishDoc;
    private JMenu menuEdit;
    private JMenu menuPreferences;
    private JMenu menuAnnotation;
    private JMenu menuTier;
    private JMenu menuType;
    private JMenu menuWindow;
    private ButtonGroup windowsGroup;
    private JMenu menuSearch;
    private JMenu menuView;
    private JMenu menuOptions;
    private JCheckBoxMenuItem menuItemNativeMedia;
    private JCheckBoxMenuItem menuItemPermanentDetached;
    private JCheckBoxMenuItem menuMacNativeLF;
    private JRadioButtonMenuItem menuItemAnnoMode;
    private JRadioButtonMenuItem menuItemSyncMode;
    private JCheckBoxMenuItem menuItemKioskMode;
    private JMenuItem menuItemPlayAround;
    private JMenuItem menuItemRateVol;
    private JMenu menuP2P;
    private JMenu menuHelp;
    private JMenu menuFrameLength;
    private JMenu menuAppLanguage;
    private ButtonGroup languageBG;
    private JMenu menuChangeTimePropMode;
    private JMenuItem menuItemOptionsEnglish;
    private JMenuItem menuItemOptionsDutch;
    private JMenuItem menuItemScrubTrans;
    private ElanLayoutManager layoutManager;
    private ViewerManager2 viewerManager;
    private ElanP2P elanP2P;

    /**
     * The no arg constructor creates an empty elan frame containing a menubar
     * with a limited set of menu items.
     */
    public ElanFrame2() {
        /* disabled for the time being
        String p2p = System.getProperty("P2P");

        if ((p2p != null) && p2p.equals("true")) {
            elanP2P = new ElanP2P(this);
        }
        */

        // set the initial title
        setTitle("Elan");

        ImageIcon icon = new ImageIcon(this.getClass().getResource("/mpi/eudico/client/annotator/resources/ELAN16.png"));

        if (icon != null) {
            setIconImage(icon.getImage());
        } else {
            setIconImage(null);
        }

        initFrame();
    }

    /**
     * Constructor that accepts the path to an .eaf file. This first creates an
     * empty frame and then calls openEAF to open the specified .eaf file.
     *
     * @param path the location of the eaf file
     *
     * @see #openEAF(String)
     */
    public ElanFrame2(final String path) {
        this();

        if (path != null) {
            SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        openEAF(path);
                    }
                });
        }
    }

    /**
     * Creates a new ElanFrame2 instance
     *
     * @param eafPath DOCUMENT ME!
     * @param mediaFiles DOCUMENT ME!
     */
    public ElanFrame2(final String eafPath, final Vector mediaFiles) {
        this();

        if (eafPath != null) {
            SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        openEAF(eafPath, mediaFiles);
                    }
                });
        }
    }

    /**
     * Constructor that accepts a Transcription object to use for the current
     * ElanFrame.
     *
     * @param transcription the trancription
     */
    public ElanFrame2(Transcription transcription) {
        this();
        transcriptionForThisFrame = transcription;

        // fill the frame if transcription != null
        if (transcriptionForThisFrame != null) {
            try {
                //new InitThread(transcriptionForThisFrame.getName()).start();
                initElan();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Open an .eaf file defined by a full path string
     * Nov 2006: changed access from private to package
     * @param fullPath
     * @param mediaFiles a vector containing associated media files
     */
    void openEAF(String fullPath, Vector mediaFiles) {
        File fileTemp = new File(fullPath);

        //check if file exists and is a file
        if (!fileTemp.exists() || fileTemp.isDirectory()) {
            String strMessage = ElanLocale.getString("Menu.Dialog.Message1");
            strMessage += fullPath;
            strMessage += ElanLocale.getString("Menu.Dialog.Message2");

            String strError = ElanLocale.getString("Message.Error");
            JOptionPane.showMessageDialog(this, strMessage, strError,
                JOptionPane.ERROR_MESSAGE);

            return;
        }

        //check if file is a '.eaf' file
        if (fileTemp.toString().toLowerCase().endsWith(".eaf") == false) {
            String strMessage = ElanLocale.getString("Menu.Dialog.Message1");
            strMessage += fullPath;
            strMessage += ElanLocale.getString("Menu.Dialog.Message3");

            String strError = ElanLocale.getString("Message.Error");
            JOptionPane.showMessageDialog(this, strMessage, strError,
                JOptionPane.ERROR_MESSAGE);

            return;
        }

        //open the eaf and get the Transcription from it
        try {
            // When config files are possible check if eaf or configuration file
            //           String path = chooser.getSelectedFile().getAbsolutePath();
            //String path = fullPath;
            String path = fileTemp.getAbsolutePath();

            // replace all backslashes by forward slashes
            path = path.replace('\\', '/');

            //long before = System.currentTimeMillis();
            Transcription transcription = new TranscriptionImpl(new File(path).getAbsolutePath());

            //long after = System.currentTimeMillis();
            //System.out.println("open eaf took " + (after - before) + " ms");
            transcription.setUnchanged();

            if (mediaFiles != null) {
                //Vector descriptors = createMediaDescriptors(mediaFiles);
                Vector descriptors = MediaDescriptorUtil.createMediaDescriptors(mediaFiles);

                // improve this; check and compare with the medianames from the eaf
                transcription.setMediaDescriptors(descriptors);
                transcription.setChanged();
            }

            int lastSlash = path.lastIndexOf('/');
            String eafPath = path.substring(0, lastSlash);
            boolean validMedia = checkMedia(transcription, eafPath);

            if (!validMedia) {
                // ask if incomplete media session is ok, if not return
                int answer = JOptionPane.showConfirmDialog(this,
                        ElanLocale.getString(
                            "Frame.ElanFrame.IncompleteMediaQuestion"),
                        ElanLocale.getString(
                            "Frame.ElanFrame.IncompleteMediaAvailable"),
                        JOptionPane.YES_NO_OPTION);

                if (answer != JOptionPane.YES_OPTION) {
                    return;
                }
            }

            if (transcriptionForThisFrame != null) {
                // create a new ElanFrame for the Transcription
                //new ElanFrame2(transcription);
                FrameManager.getInstance().createFrame(transcription);
            } else {
                transcriptionForThisFrame = transcription;

                //new InitThread(transcriptionForThisFrame.getName()).start();
                initElan();

                FrameManager.getInstance().updateFrameTitle(this,
                    transcription.getFullPath());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @param fullPath DOCUMENT ME!
     */
    public void openEAF(String fullPath) {
        openEAF(fullPath, null);
    }

    /**
     * Sets the transcription (document) for this frame. Only effective if this
     * frame is an empty frame, if there already is an transcription this
     * method simply returns.
     *
     * @param transcription the transcription object
     */
    void setTranscription(Transcription transcription) {
        if ((transcriptionForThisFrame != null) ||
                !(transcription instanceof TranscriptionImpl)) {
            return;
        }

        transcriptionForThisFrame = transcription;

        initElan();
    }

    /**
     * Returns the viewermanager for this frame
     *
     * @return DOCUMENT ME!
     */
    public ViewerManager2 getViewerManager() {
        return viewerManager;
    }

    /**
     * Check for existence of media files and fix URLs if needed and possible.
     * The directory from which the .eaf file came is used as an alternative
     * location for the media file if the absolute path does not exist. When
     * the file is not found the user is prompted with a file chooser to
     * locate the file.
     *
     * @param transcription
     * @param eafPath DOCUMENT ME!
     *
     * @return boolean that flags if the media descriptors are valid
     */
    public boolean checkMedia(Transcription transcription, String eafPath) {
        boolean validMedia = true;
        File currentDirectory = null;

        // make sure the eaf path is treated the same way as media files,
        // i.e. it starts with file:/// or file://
        String fullEAFURL = FileUtility.pathToURLString(transcription.getFullPath());

        try {
            Vector mediaDescriptors = transcription.getMediaDescriptors();

            for (int i = 0; i < mediaDescriptors.size(); i++) {
                MediaDescriptor md = (MediaDescriptor) mediaDescriptors.elementAt(i);

                // remove the file: part of the URL, leading slashes are no problem
                int colonPos = md.mediaURL.indexOf(':');

                // wwj: if the url begins with rtsp, bypass the file checking
                if (colonPos > 0) {
                    String urlhead = md.mediaURL.substring(0, colonPos);

                    if (urlhead.trim().equalsIgnoreCase("rtsp")) {
                        continue;
                    }
                }

                String fileName = md.mediaURL.substring(colonPos + 1);

                // replace all back slashes by forward slashes
                fileName = fileName.replace('\\', '/');

                File file = new File(fileName);

                if (!file.exists()) {
                    // look for the file in the local directory
                    int lastSlashPos = fileName.lastIndexOf('/');
                    String localFileName = fileName.substring(lastSlashPos + 1);
                    file = new File(eafPath + "/" + localFileName);

                    if (file.exists()) {
                        // adjust urls
                        adjustMediaDescriptors(mediaDescriptors, i,
                            file.getAbsolutePath());
                        transcription.setChanged();

                        continue;
                    }

                    // look in the relative path stored in the mediadescriptor
                    if (md.relativeMediaURL != null) {
                        String relUrl = md.relativeMediaURL;

                        if (relUrl.startsWith("file:/")) {
                            relUrl = relUrl.substring(6);
                        }

                        // resolve relative url and check location
                        String absPath = FileUtility.getAbsolutePath(fullEAFURL,
                                relUrl);

                        if (absPath != null) {
                            file = new File(absPath);

                            if (file.exists()) {
                                adjustMediaDescriptors(mediaDescriptors, i,
                                    file.getAbsolutePath());
                                transcription.setChanged();

                                continue;
                            }
                        }
                    }

                    // look in a relative path ../Media
                    file = new File(eafPath + "/../Media/" + localFileName);

                    if (file.exists()) {
                        // adjust urls
                        adjustMediaDescriptors(mediaDescriptors, i,
                            file.getAbsolutePath());
                        transcription.setChanged();

                        continue;
                    }

                    // look in a relative path ../media
                    file = new File(eafPath + "/../media/" + localFileName);

                    if (file.exists()) {
                        // adjust urls
                        adjustMediaDescriptors(mediaDescriptors, i,
                            file.getAbsolutePath());
                        transcription.setChanged();

                        continue;
                    }

                    // no fallback worked, prompt the user to locate the file
                    JFileChooser chooser = new JFileChooser();

                    if (md.mimeType.equals(MediaDescriptor.WAV_MIME_TYPE)) {
                        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                                ElanFileFilter.WAV_TYPE));
                    } else if (md.mimeType.equals(MediaDescriptor.MPG_MIME_TYPE)) {
                        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                                ElanFileFilter.MPEG_TYPE));
                    } else {
                        FileFilter mediaFilter = ElanFileFilter.createFileFilter(ElanFileFilter.MEDIA_TYPE);
                        chooser.addChoosableFileFilter(mediaFilter);
                        chooser.addChoosableFileFilter(ElanFileFilter.createFileFilter(
                                ElanFileFilter.MP4_TYPE));
                        chooser.addChoosableFileFilter(ElanFileFilter.createFileFilter(
                                ElanFileFilter.QT_TYPE));
                        chooser.setFileFilter(mediaFilter);
                    }

                    chooser.setDialogTitle(ElanLocale.getString(
                            "Frame.ElanFrame.LocateMedia") + ": " +
                        localFileName);

                    boolean found = false;

                    while (!found) {
                        if (currentDirectory != null) {
                            chooser.setCurrentDirectory(currentDirectory);
                        }

                        chooser.setSelectedFile(new File(file.getName()));

                        int returnVal = chooser.showOpenDialog(this);

                        if (returnVal == JFileChooser.APPROVE_OPTION) {
                            currentDirectory = chooser.getCurrentDirectory();

                            //String name = chooser.getSelectedFile().getName();
                            //if (name.equals(localFileName)) {
                            if (!chooser.getSelectedFile().exists()) {
                                JOptionPane.showMessageDialog(null,
                                    ElanLocale.getString(
                                        "Frame.ElanFrame.LocateMedia"),
                                    ElanLocale.getString("Message.Error"),
                                    JOptionPane.ERROR_MESSAGE);
                                found = false;

                                continue;
                            }

                            // adjust urls
                            adjustMediaDescriptors(mediaDescriptors, i,
                                chooser.getSelectedFile().getAbsolutePath());
                            transcription.setChanged();

                            found = true;

                            /*} else {
                               // we do not allow another media file name than the original
                               JOptionPane.showMessageDialog(null,
                                   ElanLocale.getString(
                                       "Frame.ElanFrame.MediaFileRenamed"),
                                   ElanLocale.getString("Message.Error"),
                                   JOptionPane.ERROR_MESSAGE);
                               }
                             */
                        } else {
                            // the user did choose cancel and thereby gave up locating the file
                            md.isValid = false;
                            validMedia = false;

                            break;
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        return validMedia;
    }

    /**
     * Replace the current path in a media descriptor and all the media
     * descriptors that are derived from it
     *
     * @param mediaDescriptors
     * @param index position in the Vector of descriptors of the descriptor
     *        which media URL must be changed
     * @param newPath the new file path that must become the new media URL
     */
    private void adjustMediaDescriptors(Vector mediaDescriptors, int index,
        String newPath) {
        // remember the old URL
        String oldURL = ((MediaDescriptor) mediaDescriptors.elementAt(index)).mediaURL;

        //String newURL = pathToURLString(newPath);
        String newURL = FileUtility.pathToURLString(newPath);
        String newExt = null;

        if (newURL.indexOf('.') > -1) {
            newExt = newURL.substring(newURL.lastIndexOf('.') + 1);
        }

        // replace the old URL and mime type
        ((MediaDescriptor) mediaDescriptors.elementAt(index)).mediaURL = newURL;
        ((MediaDescriptor) mediaDescriptors.elementAt(index)).mimeType = MediaDescriptorUtil.mimeTypeForExtension(newExt);

        // replace the extracted from URL's
        for (int i = 0; i < mediaDescriptors.size(); i++) {
            String extractedFrom = ((MediaDescriptor) mediaDescriptors.elementAt(i)).extractedFrom;

            if (oldURL.equals(extractedFrom)) {
                ((MediaDescriptor) mediaDescriptors.elementAt(i)).extractedFrom = newURL;
            }
        }
    }

    /**
     * Init Elan for a Transcription mediaDecriptors should be contained in
     * Transcription Is this the place to create all the viewers, must there
     * be getters for these viewers who needs to know about them?
     */
    private void initElan() {
        setTitle("Initializing....");

        viewerManager = new ViewerManager2(transcriptionForThisFrame);
        layoutManager = new ElanLayoutManager(this, viewerManager);
        ELANCommandFactory.addDocument(this, viewerManager, layoutManager);
        MediaDescriptorUtil.createMediaPlayers((TranscriptionImpl) transcriptionForThisFrame,
            transcriptionForThisFrame.getMediaDescriptors());

        ElanMediaPlayerController mediaController = viewerManager.getMediaPlayerController();
        layoutManager.add(mediaController);

        TimeLineViewer timeLineViewer = viewerManager.createTimeLineViewer();
        layoutManager.add(timeLineViewer);

        InterlinearViewer interlinearViewer = viewerManager.createInterlinearViewer();
        layoutManager.add(interlinearViewer);

        layoutManager.showTimeLineViewer();

        GridViewer gridViewer = viewerManager.createGridViewer();
        layoutManager.add(gridViewer);

        TextViewer textViewer = viewerManager.createTextViewer();
        layoutManager.add(textViewer);

        SubtitleViewer subtitleViewer1 = viewerManager.createSubtitleViewer();
        subtitleViewer1.setViewerIndex(1);
        layoutManager.add(subtitleViewer1);

        SubtitleViewer subtitleViewer2 = viewerManager.createSubtitleViewer();
        subtitleViewer2.setViewerIndex(2);
        layoutManager.add(subtitleViewer2);

        SubtitleViewer subtitleViewer3 = viewerManager.createSubtitleViewer();
        subtitleViewer3.setViewerIndex(3);
        layoutManager.add(subtitleViewer3);

        SubtitleViewer subtitleViewer4 = viewerManager.createSubtitleViewer();
        subtitleViewer4.setViewerIndex(4);
        layoutManager.add(subtitleViewer4);

        if (transcriptionForThisFrame.getLinkedFileDescriptors().size() > 0) {
            LinkedFileDescriptorUtil.initLinkedFiles((TranscriptionImpl) transcriptionForThisFrame);
        }

        //		layoutManager.add(new mpi.eudico.p2p.CollaborationPanel());
        if (elanP2P != null) {
            elanP2P.setManagers(viewerManager, layoutManager);
        }

        // temp block for svg
        if (SVGPrefs.getUseSVG() ||
                (((TranscriptionImpl) transcriptionForThisFrame).getSVGFile() != null)) {
            // lightweight svgviewer
            if (viewerManager.getMasterMediaPlayer() instanceof JMFGraphicMediaPlayer) {
                // don't need to add to layout
                menuItemFileOpen.setEnabled(false);
                menuItemFileNew.setEnabled(false);
                menuImport.setEnabled(false);
            }
        }

        ElanLocale.addElanLocaleListener(transcriptionForThisFrame, this);

        setFrameTitle();

        initMenusAndCommands();

        if (viewerManager.getMasterMediaPlayer().isFrameRateAutoDetected()) {
            // disable the menu to change the video standard, i.e. the frame length
            setMenuEnabled(FrameConstants.FRAME_LENGTH, false);
        }

        Preferences.addPreferencesListener(transcriptionForThisFrame,
            layoutManager);
        Preferences.addPreferencesListener(transcriptionForThisFrame, this);
        Preferences.notifyListeners(transcriptionForThisFrame);

        // a few prefs to load only on load
        loadPreferences();

        // instantiate Viewer components:
        //  use CommandActions for relevant toolbar buttons and combobox menu items
        //  position and size components using LayoutManager
        layoutManager.doLayout();

        //  this sucks but is needed to ensure visible video on the mac
        viewerManager.getMasterMediaPlayer().setMediaTime(0);
    }

    private void setFrameTitle() {
        try {
            if (transcriptionForThisFrame != null) {
                if (transcriptionForThisFrame.getName().equals(TranscriptionImpl.UNDEFINED_FILE_NAME)) {
                    setTitle("Elan - " +
                        ElanLocale.getString(
                            "Frame.ElanFrame.UndefinedFileName"));
                } else {
                    setTitle("Elan - " + transcriptionForThisFrame.getName());
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Performs some general initialization for an empty frame.
     */
    private void initFrame() {
        Locale savedLocale = (Locale) Preferences.get("Locale", null);

        if (savedLocale != null) {
            ElanLocale.setLocale(savedLocale);
        }

        // create the initial menu items
        initMenuBar();

        // listen for WindowEvents events on the ElanFrame
        addWindowListener(new ElanFrameWindowListener());

        // require the program to handle the operation in the
        // windowClosing method of a registered WindowListener object.
        setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);

        // add this elanframe to locale listener
        //ElanLocale.addElanLocaleListener(this);
        pack();

        /*
           Dimension dimSize = new Dimension(800, 650);
           setSize((int)dimSize.getWidth(), (int)dimSize.getHeight());
           Dimension dimLocation = Toolkit.getDefaultToolkit().getScreenSize();
           setLocation((int)(dimLocation.getWidth()/2-getWidth()/2),(int)(dimLocation.getHeight()/2-getHeight()/2));
         */
        Dimension d = (Dimension) Preferences.get("FrameSize", null);

        if (d != null) {
            setSize(d);
        } else {
            Dimension dimSize = new Dimension(800, 650);
            setSize((int) dimSize.getWidth(), (int) dimSize.getHeight());
        }

        Point p = (Point) Preferences.get("FrameLocation", null);

        if (p != null) {
            setLocation(p);
        } else {
            Dimension dimLocation = Toolkit.getDefaultToolkit().getScreenSize();
            setLocation((int) ((dimLocation.getWidth() / 2) - (getWidth() / 2)),
                (int) ((dimLocation.getHeight() / 2) - (getHeight() / 2)));
        }

        // the call to setVisible should be moved to (a new thread created in) 
        // the class that called a constructor....
        setVisible(true);
    }

    private void initMenuBar() {
        menuBar = new JMenuBar();

        setJMenuBar(menuBar);

        //make menu visible / appear above heavyweight video
        JPopupMenu.setDefaultLightWeightPopupEnabled(false);

        MenuAction ma;
        ma = new MenuAction("Menu.File");
        menuFile = new JMenu(ma);
        menuActions.add(ma);
        menuBar.add(menuFile);

        ma = new NewMA("Menu.File.New", this);
        menuActions.add(ma);
        menuItemFileNew = new JMenuItem(ma);
        menuFile.add(menuItemFileNew);

        ma = new OpenMA("Menu.File.Open", this);
        menuActions.add(ma);
        menuItemFileOpen = new JMenuItem(ma);
        menuFile.add(menuItemFileOpen);

        ma = new MenuAction("Menu.File.OpenRecent");
        menuActions.add(ma);
        menuRecentFiles = new JMenu(ma);
        menuFile.add(menuRecentFiles);
        menuFile.addSeparator();

        // temp item, remove before release.
        /*
        ma = new MultiEAFScrubberMA("Menu.File.ScrubTranscriptions", this);
        menuActions.add(ma);
        menuItemScrubTrans = new JMenuItem(ma);
        menuFile.add(menuItemScrubTrans);
        */
        ma = new MenuAction("Menu.File.Export.MultipleFiles");
        menuActions.add(ma);

        JMenu exportMenuMulti = new JMenu(ma);
        menuFile.add(exportMenuMulti);

        ma = new ExportTabMultiMA("Menu.File.Export.Tab", this);
        menuActions.add(ma);
        exportMenuMulti.add(new JMenuItem(ma));

        ma = new ExportAnnotationsMultiMA("Menu.File.Export.AnnotationListMulti",
                this);
        menuActions.add(ma);
        exportMenuMulti.add(new JMenuItem(ma));

        ma = new ExportWordsMultiMA("Menu.File.Export.WordList", this);
        menuActions.add(ma);
        exportMenuMulti.add(new JMenuItem(ma));

        ma = new MenuAction("Menu.File.Import");
        menuActions.add(ma);
        menuImport = new JMenu(ma);
        menuFile.add(menuImport);

        ma = new ImportShoeboxMA("Menu.File.Import.Shoebox", this);
        menuActions.add(ma);
        menuItemShoeboxImport = new JMenuItem(ma);
        menuImport.add(menuItemShoeboxImport);

        ma = new ImportCHATMA("Menu.File.Import.CHAT", this);
        menuActions.add(ma);
        menuItemCHATImport = new JMenuItem(ma);
        menuImport.add(menuItemCHATImport);

        ma = new ImportTranscriberMA("Menu.File.Import.Transcriber", this);
        menuActions.add(ma);
        menuItemTranscriberImport = new JMenuItem(ma);
        menuImport.add(menuItemTranscriberImport);

        ma = new ImportDelimitedTextMA("Menu.File.Import.Delimited", this);
        menuActions.add(ma);
        menuImport.add(new JMenuItem(ma));

        menuFile.addSeparator();

        ma = new ExitMA("Menu.File.Exit");
        menuActions.add(ma);
        menuItemFileExit = new JMenuItem(ma);
        menuFile.add(menuItemFileExit);

        ma = new MenuAction("Menu.Edit");
        menuActions.add(ma);
        menuEdit = new JMenu(ma);
        menuBar.add(menuEdit);
        menuEdit.setVisible(false);

        ma = new MenuAction("Menu.Annotation");
        menuActions.add(ma);
        menuAnnotation = new JMenu(ma);
        menuBar.add(menuAnnotation);
        menuAnnotation.setVisible(false);

        ma = new MenuAction("Menu.Tier");
        menuActions.add(ma);
        menuTier = new JMenu(ma);
        menuBar.add(menuTier);
        menuTier.setVisible(false);

        ma = new MenuAction("Menu.Type");
        menuActions.add(ma);
        menuType = new JMenu(ma);
        menuBar.add(menuType);
        menuType.setVisible(false);

        ma = new MenuAction("Menu.Search");
        menuActions.add(ma);
        menuSearch = new JMenu(ma);
        menuBar.add(menuSearch);
        menuSearch.setVisible(false);

        ma = new MenuAction("Menu.View");
        menuActions.add(ma);
        menuView = new JMenu(ma);
        menuBar.add(menuView);
        menuView.setVisible(false);

        ma = new MenuAction("Menu.Options");
        menuActions.add(ma);
        menuOptions = new JMenu(ma);
        menuBar.add(menuOptions);

        ma = new MenuAction("Menu.Window");
        menuActions.add(ma);
        menuWindow = new JMenu(ma);
        windowsGroup = new ButtonGroup();
        menuBar.add(menuWindow);

        ma = new MenuAction("Menu.Help");
        menuActions.add(ma);
        menuHelp = new JMenu(ma);
        menuBar.add(menuHelp);

        //menuHelp.setVisible(false);
        ma = new AboutMA("Menu.Help.About", this);
        menuActions.add(ma);
        menuHelp.add(new JMenuItem(ma));

        // temporary and therefore old fashioned menu item to allow fallback to JMF media players
        // on windows machines. As soon as native media is succesfully used in a few releases
        // this menu item can be deleted. The only action performed while toggling this item is
        // setting the PreferredMediaFramework property that is used by the player factory
        if (System.getProperty("os.name").toLowerCase().indexOf("windows") > -1) {
            menuItemNativeMedia = new JCheckBoxMenuItem();
            menuItemNativeMedia.setText("Use Native Media Platform");
            menuItemNativeMedia.addActionListener(this);

            // haal de state uit de property
            boolean nativePrefered = true;
            String preferredMF = System.getProperty("PreferredMediaFramework");

            // set here default choice if nothing defined
            if (preferredMF == null) {
                System.setProperty("PreferredMediaFramework", "NativeWindows");

                //System.setProperty("PreferredMediaFramework", "JMF");
            } else if (!preferredMF.equals("NativeWindows")) {
                nativePrefered = false;
            }

            menuItemNativeMedia.setState(nativePrefered);

            // optionally suppress jmf support for the standalone version
            String suppressJMF = System.getProperty("suppressJMF");

            if ((suppressJMF == null) ||
                    !suppressJMF.toLowerCase().equals("true")) {
                menuOptions.add(menuItemNativeMedia);
            }
        }

        if (elanP2P != null) {
            ma = new MenuAction("Menu.P2P");
            menuActions.add(ma);
            menuP2P = new JMenu(ma);
            menuBar.add(menuP2P);

            menuItemDiscoverDoc = new JMenuItem();
            menuItemDiscoverDoc.addActionListener(this);
            menuItemDiscoverDoc.setActionCommand("DiscoverDoc");
            menuP2P.add(menuItemDiscoverDoc);
        }

        if (System.getProperty("os.name").startsWith("Mac OS")) {
            // inserted by AR to allow choice for permanent detached video at startup
            menuItemPermanentDetached = new JCheckBoxMenuItem();
            menuItemPermanentDetached.setText("Use detached media window");
            menuItemPermanentDetached.addActionListener(this);

            // haal de state uit de preferences
            Boolean permanentDetached = (Boolean) Preferences.get("PreferredMediaWindow",
                    null);

            if (permanentDetached == null) {
                permanentDetached = new Boolean(false); // default usage is attached media window
            }

            menuItemPermanentDetached.setState(permanentDetached.booleanValue());
            menuOptions.add(menuItemPermanentDetached);

            // end of insertion by AR
            // Mac L&F
            menuMacNativeLF = new JCheckBoxMenuItem();
            menuMacNativeLF.setText("Use Mac Look And Feel");
            menuMacNativeLF.setActionCommand("MacNativeLF");
            menuMacNativeLF.addActionListener(this);

            Boolean macNativePref = (Boolean) Preferences.get("UseMacLF", null);
            menuOptions.add(menuMacNativeLF);

            if ((macNativePref != null) && !macNativePref.booleanValue()) {
                menuMacNativeLF.setState(false);

                try {
                    UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
                    SwingUtilities.updateComponentTreeUI(this);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            } else {
                menuMacNativeLF.setState(true);
            }

            // use reflection to make this compile on all systems
            try {
                Class macHandler = Class.forName(
                        "mpi.eudico.client.mac.MacAppHandler");
                Class macList = Class.forName(
                        "mpi.eudico.client.mac.MacApplicationListener");
                java.lang.reflect.Constructor con = macHandler.getConstructor(new Class[] {
                            macList
                        });
                con.newInstance(new Object[] { this });
            } catch (Exception ex) {
                System.out.println("Could not load Mac application handler.");
            }
        }

        updateLocale();
    }

    /**
     * DOCUMENT ME!
     *
     * @param e DOCUMENT ME!
     */
    public void actionPerformed(ActionEvent e) {
        String command = e.getActionCommand();

        if (command.equals("DiscoverDoc")) {
            if (elanP2P != null) {
                Command c = ELANCommandFactory.createCommand(null,
                        ELANCommandFactory.DISCOVER_DOC);
                c.execute(elanP2P, new Object[] { this });
            }
        } else if (command.equals("PublishDoc")) {
            if (elanP2P != null) {
                Command c = ELANCommandFactory.createCommand(null,
                        ELANCommandFactory.PUBLISH_DOC);
                c.execute(elanP2P,
                    new Object[] { this, transcriptionForThisFrame });
            }
        } else if (command.equals("Use Native Media Platform")) {
            boolean useNativeMedia = menuItemNativeMedia.getState();

            if (useNativeMedia) {
                System.out.println(
                    "Setting preferred media framework to Native");
                System.setProperty("PreferredMediaFramework", "NativeWindows");
            } else {
                System.out.println("Setting preferred media framework to JMF");
                System.setProperty("PreferredMediaFramework", "JMF");
            }
        } else if (command.equals("Use detached media window")) {
            Preferences.set("PreferredMediaWindow",
                new Boolean(menuItemPermanentDetached.getState()), null);
        } else if (command.equals("MacNativeLF")) {
            switchMacLF();
        }
    }

    private void initMenusAndCommands() {
        // instantiate CommandActions, also UndoCA and RedoCA

        /* sample:
           CommandAction playSelectionCA = ELANCommandFactory.getCommandAction(ELANCommandFactory.PLAY_SELECTION);
         */

        // instantiate JMenuItems, where possible with CommandActions as args for constructor

        /* sample:
           playSelectionItem = new JMenuItem(playSelectionCA);
           playSelectionItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, ActionEvent.CTRL_MASK));
           menuInvisible.add(playSelectionItem);   // add to menu, if only for button then to invisible menu
         */

        // add actions with accelerator keys and without a menu item to the input
        // and action map
        InputMap inputMap = menuBar.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        ActionMap actionMap = menuBar.getActionMap();

        if (inputMap instanceof ComponentInputMap && (actionMap != null)) {
            Action[] invActions = new Action[] {
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PLAY_PAUSE),
                    

                    //ELANCommandFactory.getCommandAction(transcriptionForThisFrame, ELANCommandFactory.NEW_ANNOTATION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.GO_TO_BEGIN),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.GO_TO_END),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.SECOND_LEFT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.SECOND_RIGHT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PLAY_SELECTION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PLAY_AROUND_SELECTION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PREVIOUS_ANNOTATION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.NEXT_ANNOTATION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.ANNOTATION_UP),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.ANNOTATION_DOWN),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PREVIOUS_SCROLLVIEW),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.NEXT_SCROLLVIEW),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PREVIOUS_FRAME),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.NEXT_FRAME),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PIXEL_LEFT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PIXEL_RIGHT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.CLEAR_SELECTION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.SELECTION_BOUNDARY),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.SELECTION_MODE),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.LOOP_MODE),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.MODIFY_ANNOTATION_TIME),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PLAYBACK_RATE_TOGGLE),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PLAYBACK_VOLUME_TOGGLE),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.NEXT_ACTIVE_TIER),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.PREVIOUS_ACTIVE_TIER),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.NEW_ANNOTATION_ALT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.MODIFY_ANNOTATION_ALT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.DELETE_ANNOTATION_ALT),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.KEY_CREATE_ANNOTATION),
                    ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                        ELANCommandFactory.COPY_TO_NEXT_ANNOTATION),
                    new NextWindowMA("Menu.Window.Next"),
                    new PrevWindowMA("Menu.Window.Previous")
                };

            String id = "Act-";
            String nextId;

            for (int i = 0; i < invActions.length; i++) {
                nextId = id + i;
                inputMap.put((KeyStroke) invActions[i].getValue(
                        Action.ACCELERATOR_KEY), nextId);
                actionMap.put(nextId, invActions[i]);
                registeredActions.put(nextId, invActions[i]);
            }
        }

        MenuAction ma;

        // temp
        if (menuItemScrubTrans != null) {
            menuFile.remove(menuItemScrubTrans);
        }

        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.CLOSE)), 3);
        menuFile.insertSeparator(4);
        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SAVE)), 5);
        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SAVE_AS)), 6);
        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.SAVE_AS_TEMPLATE)), 7);
        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.SAVE_SELECTION_AS_EAF)), 8);
        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.MERGE_TRANSCRIPTIONS)), 9);

        ma = new MenuAction("Menu.File.Backup.Auto");
        menuActions.add(ma);
        menuBackup = new JMenu(ma);
        menuFile.add(menuBackup, 10);

        ButtonGroup backupGroup = new ButtonGroup();

        // retrieve the stored value for backup interval
        Integer buDelay = (Integer) Preferences.get("BackUpDelay", null);
        JRadioButtonMenuItem neverMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_NEVER));

        if ((buDelay == null) ||
                ((buDelay != null) &&
                (buDelay.compareTo(Constants.BACKUP_NEVER) == 0))) {
            neverMI.setSelected(true);
        }

        JRadioButtonMenuItem backup1MI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_1));

        if ((buDelay != null) && (buDelay.compareTo(Constants.BACKUP_1) == 0)) {
            backup1MI.setSelected(true);
        }

        JRadioButtonMenuItem backup5MI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_5));

        if ((buDelay != null) && (buDelay.compareTo(Constants.BACKUP_5) == 0)) {
            backup5MI.setSelected(true);
        }

        JRadioButtonMenuItem backup10MI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_10));

        if ((buDelay != null) && (buDelay.compareTo(Constants.BACKUP_10) == 0)) {
            backup10MI.setSelected(true);
        }

        JRadioButtonMenuItem backup20MI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_20));

        if ((buDelay != null) && (buDelay.compareTo(Constants.BACKUP_20) == 0)) {
            backup20MI.setSelected(true);
        }

        JRadioButtonMenuItem backup30MI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BACKUP_30));

        if ((buDelay != null) && (buDelay.compareTo(Constants.BACKUP_30) == 0)) {
            backup30MI.setSelected(true);
        }

        backupGroup.add(neverMI);
        backupGroup.add(backup1MI);
        backupGroup.add(backup5MI);
        backupGroup.add(backup10MI);
        backupGroup.add(backup20MI);
        backupGroup.add(backup30MI);
        menuBackup.add(neverMI);
        menuBackup.add(backup1MI);
        menuBackup.add(backup5MI);
        menuBackup.add(backup10MI);
        menuBackup.add(backup20MI);
        menuBackup.add(backup30MI);

        menuFile.insertSeparator(11);

        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.PAGESETUP)),
            12);

        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.PREVIEW)), 13);

        menuFile.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.PRINT)), 14);

        ma = new MenuAction("Menu.File.Export");
        menuActions.add(ma);
        menuExport = new JMenu(ma);
        menuFile.add(menuExport, 16);

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_SHOEBOX)));
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_TOOLBOX)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_CHAT)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_TAB)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_TIGER)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_INTERLINEAR)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_HTML)));
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_TRAD_TRANSCRIPT)));
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_PRAAT_GRID)));
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_WORDS)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_SMIL)));

        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_QT_SUB)));
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_SUBTITLES)));

        //		menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
        //				 transcriptionForThisFrame, ELANCommandFactory.EXPORT_TEX)));
        // maybe test here if available and only show menu item if available:
        //
        //ExportMediaCA ca = (ExportMediaCA) ELANCommandFactory.getCommandAction(transcriptionForThisFrame, ELANCommandFactory.EXPORT_MEDIA);
        //if (ca.isAvailable()) {
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EXPORT_MEDIA)));

        //}
        menuExport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_IMAGE_FROM_WINDOW)));
        menuImport.addSeparator();
        menuImport.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.IMPORT_PRAAT_GRID)));

        // enable menu's
        menuEdit.setVisible(true);

        // menu items with command actions
        CommandAction undoCA = ELANCommandFactory.getUndoCA(transcriptionForThisFrame);
        ElanMenuItem undoItem = new ElanMenuItem(undoCA);

        menuEdit.add(undoItem);

        CommandAction redoCA = ELANCommandFactory.getRedoCA(transcriptionForThisFrame);
        ElanMenuItem redoItem = new ElanMenuItem(redoCA);

        menuEdit.add(redoItem);
        menuEdit.addSeparator();

        menuEdit.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.EDIT_CV_DLG)));

        menuEdit.addSeparator();
        menuEdit.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.LINKED_FILES_DLG)));
        menuEdit.addSeparator();

        ma = new MenuAction("Menu.Edit.Preferences");
        menuActions.add(ma);
        menuPreferences = new JMenu(ma);
        menuEdit.add(menuPreferences);
        menuPreferences.add(new ElanMenuItem(
                ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                    ELANCommandFactory.IMPORT_PREFS)));
        menuPreferences.add(new ElanMenuItem(
                ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                    ELANCommandFactory.EXPORT_PREFS)));

        menuAnnotation.setVisible(true);

        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.NEW_ANNOTATION)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.NEW_ANNOTATION_BEFORE)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.NEW_ANNOTATION_AFTER)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.MODIFY_ANNOTATION)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.REMOVE_ANNOTATION_VALUE)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.DELETE_ANNOTATION)));

        if (SVGPrefs.getUseSVG() ||
                (((TranscriptionImpl) transcriptionForThisFrame).getSVGFile() != null)) {
            menuAnnotation.add(new ElanMenuItem(
                    ELANCommandFactory.getCommandAction(
                        transcriptionForThisFrame,
                        ELANCommandFactory.MODIFY_GRAPHIC_ANNOTATION)));
        }

        menuAnnotation.addSeparator();
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.COPY_ANNOTATION)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.COPY_ANNOTATION_TREE)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.DUPLICATE_ANNOTATION)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PASTE_ANNOTATION)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PASTE_ANNOTATION_TREE)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PASTE_ANNOTATION_HERE)));
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PASTE_ANNOTATION_TREE_HERE)));
        menuAnnotation.addSeparator();
        menuAnnotation.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.SHIFT_ALL_ANNOTATIONS)));

        menuTier.setVisible(true);

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.ADD_TIER)));
        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.CHANGE_TIER)));
        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.REPARENT_TIER)));
        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.DELETE_TIER)));
        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.IMPORT_TIERS)));

        menuTier.addSeparator();

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.TOKENIZE_DLG)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.FILTER_TIER_DLG)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.COPY_TIER_DLG)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.ANN_FROM_OVERLAP)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.REGULAR_ANNOTATION_DLG)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.LABEL_AND_NUMBER)));

        menuTier.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.SEGMENTATION_DLG)));

        //menuTier.addSeparator();
        menuType.setVisible(true);

        menuType.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.ADD_TYPE)));
        menuType.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.CHANGE_TYPE)));
        menuType.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.DELETE_TYPE)));
        menuType.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.IMPORT_TYPES)));

        menuSearch.setVisible(true);
        menuSearch.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SEARCH_DLG)));

        menuSearch.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.SEARCH_MULTIPLE_DLG)));
        menuSearch.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.STRUCTURED_SEARCH_MULTIPLE_DLG)));

        menuSearch.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.GOTO_DLG)));

        menuView.setVisible(true);

        ElanMenuItem menuItemViewTierDependencies = new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.TIER_DEPENDENCIES));
        menuView.add(menuItemViewTierDependencies);
        menuView.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SHORTCUTS)));
        menuView.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.FONT_BROWSER)));
        menuView.addSeparator();
        menuView.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SPREADSHEET)));
        menuView.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.STATISTICS)));

        CommandAction syntaxViewerAction = ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                ELANCommandFactory.SYNTAX_VIEWER);

        if (syntaxViewerAction != null) {
            menuView.add(new ElanMenuItem(syntaxViewerAction));
        }

        menuOptions.removeAll();
        ma = new MenuAction("Menu.Options.TimeChangePropagationMode");
        menuActions.add(ma);
        menuChangeTimePropMode = new JMenu(ma);

        ButtonGroup timePropGroup = new ButtonGroup();
        JRadioButtonMenuItem normalModeMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.TIMEPROP_NORMAL));
        JRadioButtonMenuItem bulldozerModeMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.BULLDOZER_MODE));
        JRadioButtonMenuItem shiftModeMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SHIFT_MODE));
        timePropGroup.add(normalModeMI);
        normalModeMI.setSelected(true);
        timePropGroup.add(bulldozerModeMI);
        timePropGroup.add(shiftModeMI);
        menuChangeTimePropMode.add(normalModeMI);
        menuChangeTimePropMode.add(bulldozerModeMI);
        menuChangeTimePropMode.add(shiftModeMI);
        menuOptions.add(menuChangeTimePropMode);
        menuOptions.addSeparator();

        ButtonGroup modeGroup = new ButtonGroup();
        menuItemAnnoMode = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.ANNOTATION_MODE));
        menuItemAnnoMode.setSelected(true);

        menuItemSyncMode = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SYNC_MODE));
        modeGroup.add(menuItemAnnoMode);
        modeGroup.add(menuItemSyncMode);
        menuOptions.add(menuItemAnnoMode);
        menuOptions.add(menuItemSyncMode);
        menuOptions.addSeparator();
        menuItemKioskMode = new JCheckBoxMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.KIOSK_MODE));

        //menuOptions.add(menuItemKioskMode);
        //menuOptions.addSeparator();
        menuItemPlayAround = new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PLAY_AROUND_SELECTION_DLG));
        menuOptions.add(menuItemPlayAround);
        menuItemRateVol = new ElanMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame,
                    ELANCommandFactory.PLAYBACK_TOGGLE_DLG));
        menuOptions.add(menuItemRateVol);

        menuOptions.addSeparator();

        ma = new MenuAction("Menu.Options.FrameLength");
        menuActions.add(ma);
        menuFrameLength = new JMenu(ma);

        ButtonGroup videoGroup = new ButtonGroup();
        JRadioButtonMenuItem palMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SET_PAL));
        JRadioButtonMenuItem ntscMI = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SET_NTSC));
        videoGroup.add(palMI);
        palMI.setSelected(true);
        videoGroup.add(ntscMI);
        menuFrameLength.add(palMI);
        menuFrameLength.add(ntscMI);
        menuOptions.add(menuFrameLength);
        menuOptions.addSeparator();

        ma = new MenuAction("Menu.Options.Language");
        menuActions.add(ma);
        menuAppLanguage = new JMenu(ma);
        menuOptions.add(menuAppLanguage);

        languageBG = new ButtonGroup();

        JMenuItem miCatalan = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.CATALAN));
        languageBG.add(miCatalan);
        menuAppLanguage.add(miCatalan);

        JMenuItem miGerman = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.GERMAN));
        languageBG.add(miGerman);
        menuAppLanguage.add(miGerman);

        menuItemOptionsEnglish = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.ENGLISH));
        menuItemOptionsEnglish.setSelected(true);
        languageBG.add(menuItemOptionsEnglish);
        menuAppLanguage.add(menuItemOptionsEnglish);

        JMenuItem miSpanish = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SPANISH));
        languageBG.add(miSpanish);
        menuAppLanguage.add(miSpanish);

        JMenuItem miFrench = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.FRENCH));
        languageBG.add(miFrench);
        menuAppLanguage.add(miFrench);

        menuItemOptionsDutch = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.DUTCH));
        languageBG.add(menuItemOptionsDutch);
        menuAppLanguage.add(menuItemOptionsDutch);

        JMenuItem miPortu = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.PORTUGUESE));
        languageBG.add(miPortu);
        menuAppLanguage.add(miPortu);

        JMenuItem miSwedish = new JRadioButtonMenuItem(ELANCommandFactory.getCommandAction(
                    transcriptionForThisFrame, ELANCommandFactory.SWEDISH));
        languageBG.add(miSwedish);
        menuAppLanguage.add(miSwedish);

        if (elanP2P != null) {
            menuItemPublishDoc = new JMenuItem();
            menuItemPublishDoc.addActionListener(this);
            menuItemPublishDoc.setActionCommand("PublishDoc");
            menuP2P.add(menuItemPublishDoc);

            //menuP2P.add(new ElanMenuItem(ELANCommandFactory.getCommandAction(transcriptionForThisFrame, ELANCommandFactory.PUBLISH_DOC)), 0);
        }

        updateLocale();
    }

    /**
     * DOCUMENT ME!
     */
    public void updateLocale() {
        setFrameTitle();

        // update the language menu items
        if (languageBG != null) {
            if (languageBG.getButtonCount() > 0) {
                Enumeration en = languageBG.getElements();
                Object item;
                JRadioButtonMenuItem rbItem;

                while (en.hasMoreElements()) {
                    item = en.nextElement();

                    if (item instanceof JRadioButtonMenuItem) {
                        rbItem = (JRadioButtonMenuItem) item;

                        if (ELANCommandFactory.getLocaleForKey(
                                    rbItem.getAction().getValue(Action.DEFAULT))
                                                  .equals(ElanLocale.getLocale())) {
                            rbItem.setSelected(true);

                            break;
                        }
                    }
                }
            }
        }

        if (menuP2P != null) {
            menuItemDiscoverDoc.setText(ElanLocale.getString(
                    "Menu.P2P.DiscoverDocument"));

            if (menuItemPublishDoc != null) {
                menuItemPublishDoc.setText(ElanLocale.getString(
                        "Menu.P2P.PublishDocument"));
            }
        }

        // update the special, document independent actions
        MenuAction ma;

        for (int i = 0; i < menuActions.size(); i++) {
            ma = (MenuAction) menuActions.get(i);
            ma.updateLocale();
        }
    }

    private void savePreferences() {
        // save some transcription specific preference values

        /*
        if (layoutManager != null) {
            if (layoutManager.getMultiTierControlPanel() != null) {
                List tierOrder = layoutManager.getMultiTierControlPanel()
                                                .getTierOrder();

                if (tierOrder != null) {
                    Preferences.set("TierOrder", tierOrder,
                        transcriptionForThisFrame);
                }

                String activeTierName = layoutManager.getMultiTierControlPanel()
                                                     .getActiveTierName();

                if (activeTierName != null) {
                    Preferences.set("ActiveTierName", activeTierName,
                        transcriptionForThisFrame);
                }
            }

            Preferences.set("LayoutManagerState", layoutManager.getState(),
                transcriptionForThisFrame);
        }
        */
        if (viewerManager != null) {
            Preferences.set("MediaTime",
                viewerManager.getMasterMediaPlayer().getMediaTime(),
                transcriptionForThisFrame);
            Preferences.set("SelectionBeginTime",
                viewerManager.getSelection().getBeginTime(),
                transcriptionForThisFrame);
            Preferences.set("SelectionEndTime",
                viewerManager.getSelection().getEndTime(),
                transcriptionForThisFrame);
            Preferences.set("TimeScaleBeginTime",
                viewerManager.getTimeScale().getBeginTime(),
                transcriptionForThisFrame);
        }

        // replace by setPreference();
        setPreference("Locale", ElanLocale.getLocale(), null);
        setPreference("FrameSize", getSize(), null);

        //setPreference("FrameLocation", getLocation(), null);
        Preferences.set("FrameLocation", getLocation(), null, false, true); // forces writing
    }

    private void loadPreferences() {
        // initialize some transcription specific values
        // invokeLater ensures the viewers are initialized properly before setting values
        SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    /*
                    Locale savedLocale = (Locale) Preferences.get("Locale", null);

                    if (savedLocale != null) {
                        ElanLocale.setLocale(savedLocale);
                    }
                    Object toObject = Preferences.get("TierOrder",
                            transcriptionForThisFrame);
                    Vector tierOrder;
                    if (toObject instanceof ArrayList) {
                        tierOrder = new Vector((ArrayList) toObject);
                    } else {
                        tierOrder = (Vector) toObject;
                    }

                    if (tierOrder != null) {
                        if (tierOrder.size() > 0) {
                            layoutManager.getMultiTierControlPanel()
                                         .setTierOrder(tierOrder);
                        } else if (transcriptionForThisFrame.getTiers()
                                                                .size() > 0) {
                            Vector tiers = transcriptionForThisFrame.getTiers();
                            Vector tierNames = new Vector(tiers.size());

                            for (int i = 0; i < tiers.size(); i++) {
                                Tier t = (Tier) tiers.get(i);
                                tierNames.add(t.getName());
                            }

                            layoutManager.getMultiTierControlPanel()
                                         .setTierOrder(tierNames);
                        }
                    }

                    String activeTierName = (String) Preferences.get("ActiveTierName",
                            transcriptionForThisFrame);

                    if (activeTierName != null) {
                        layoutManager.getMultiTierControlPanel()
                                     .setActiveTierForName(activeTierName);
                    }
                    */
                    Long beginTime = (Long) Preferences.get("SelectionBeginTime",
                            transcriptionForThisFrame);
                    Long endTime = (Long) Preferences.get("SelectionEndTime",
                            transcriptionForThisFrame);

                    if ((beginTime != null) && (endTime != null)) {
                        viewerManager.getSelection().setSelection(beginTime.longValue(),
                            endTime.longValue());
                    }

                    Long mediaTime = (Long) Preferences.get("MediaTime",
                            transcriptionForThisFrame);

                    if (mediaTime != null) {
                        viewerManager.getMasterMediaPlayer().setMediaTime(mediaTime.longValue());
                    }

                    Long timeScaleBeginTime = (Long) Preferences.get("TimeScaleBeginTime",
                            transcriptionForThisFrame);

                    if (timeScaleBeginTime != null) {
                        viewerManager.getTimeScale().setBeginTime(timeScaleBeginTime.longValue());
                    }

                    /*
                    HashMap layoutMap = (HashMap) Preferences.get("LayoutManagerState",
                           transcriptionForThisFrame);

                    if (layoutMap != null) {
                        layoutManager.setState(layoutMap);
                    }
                    */

                    // start the backup thread
                    Integer backupDelay = (Integer) Preferences.get("BackUpDelay",
                            null);

                    if ((backupDelay != null) && (backupDelay.intValue() > 0)) {
                        Command c = ELANCommandFactory.createCommand(transcriptionForThisFrame,
                                ELANCommandFactory.BACKUP);
                        c.execute(ELANCommandFactory.getCommandAction(
                                transcriptionForThisFrame,
                                ELANCommandFactory.BACKUP),
                            new Object[] { backupDelay });
                    }
                }
            });

        layoutManager.doLayout();
    }

    /**
     * @see PreferencesUser#setPreference(String, Object, Object)
     */
    public void setPreference(String key, Object value, Object document) {
        if (document instanceof Transcription) {
            Preferences.set(key, value, (Transcription) document, false, false);
        } else {
            Preferences.set(key, value, null, false, false);
        }
    }

    /**
     * Should these preferences be part of the loading of doc. preferences??
     * @see PreferencesListener#preferencesChanged()
     */
    public void preferencesChanged() {
        Dimension d = (Dimension) Preferences.get("FrameSize", null);

        if (d != null) {
            setSize(d);
        }

        Point p = (Point) Preferences.get("FrameLocation", null);

        if (p != null) {
            setLocation(p);
        }

        // ?? 
        Locale savedLocale = (Locale) Preferences.get("Locale", null);

        if (savedLocale != null) {
            ElanLocale.setLocale(savedLocale);
        }
    }

    /**
     * Checks whether there are any changes to save and starts the saving and or
     * closing sequence.
     *
     * @see #saveAndClose(boolean)
     * @see #doClose(boolean)
     */
    public void checkSaveAndClose() {
        if (transcriptionForThisFrame != null) {
            if (transcriptionForThisFrame.isChanged()) {
                int response = JOptionPane.showConfirmDialog(null,
                        ElanLocale.getString("Frame.ElanFrame.UnsavedData"),
                        ElanLocale.getString("Message.Warning"),
                        JOptionPane.YES_NO_CANCEL_OPTION);

                if (response == JOptionPane.YES_OPTION) {
                    saveAndClose(true);
                } else if ((response == JOptionPane.CANCEL_OPTION) ||
                        (response == JOptionPane.CLOSED_OPTION)) {
                    // the user does not want to close
                    return;
                } else {
                    doClose(true);
                }
            } else {
                doClose(true);
            }
        }
    }

    /**
     * Saves the document and starts the window closing sequence.
     * <b>Caution: </b>does not ask the user whether or not to save changes;
     * it is assumed that this has been done elsewhere.
     * Called from the Close Command or from the FrameManager.
     *
     * @param unregister if true this frame should be unregistered with the
     * FrameManager
     * @see #checkSaveAndClose()
     * @see #doClose(boolean)
     */
    public void saveAndClose(boolean unregister) {
        if (unregister) {
            FrameManager.getInstance().closeFrame(this);
        }

        if ((transcriptionForThisFrame != null) &&
                transcriptionForThisFrame.isChanged()) {
            boolean saveNewCopy = false;

            if (transcriptionForThisFrame.getName().equals(TranscriptionImpl.UNDEFINED_FILE_NAME)) {
                // save as dialog
                saveNewCopy = true;
            }

            TranscriptionStore ets = ACMTranscriptionStore.getCurrentTranscriptionStore();
            StoreCommand storeComm = new StoreCommand(ELANCommandFactory.STORE);
            storeComm.execute(transcriptionForThisFrame,
                new Object[] {
                    ets, new Boolean(false), new Boolean(saveNewCopy),
                    viewerManager.getMultiTierControlPanel().getVisibleTiers()
                });

            // HS nov 2006: check if the file actually has been saved
            // if not e.g. a Save As dialog has been canceled, return
            if (transcriptionForThisFrame.isChanged()) {
                System.out.println("Save (as) cancelled");

                return;
            }
        }

        doClose(unregister);
    }

    /**
     * Unregisteres with the FrameManager if necessary and closes
     * the frame (without saving). Performs clean up and disposes the frame.<br>
     * Called from from Close Command or from the FrameManager.<br>
     * <b>Caution: </b>does not check whether the document should be saved or not;
     * it is assumed that this has been done elsewhere.
     * @param unregister if true this frame should be unregistered with the
     * FrameManager
     * @see #checkSaveAndClose()
     * @see #saveAndClose(boolean)
     */
    public void doClose(boolean unregister) {
        if (unregister) {
            FrameManager.getInstance().closeFrame(this);
        }

        savePreferences();

        //remove document from ELANCommandFactory, unregister as listener etc.
        if (transcriptionForThisFrame != null) {
            //stop the backup task, just to be sure
            BackupCA ca = ((BackupCA) ELANCommandFactory.getCommandAction(transcriptionForThisFrame,
                    ELANCommandFactory.BACKUP));

            if (ca != null) {
                ca.stopBackUp();
            }

            ELANCommandFactory.removeDocument(viewerManager);
            Preferences.removeDocument(transcriptionForThisFrame);

            // remove this elan frame as locale listener
            ElanLocale.removeElanLocaleListener(transcriptionForThisFrame);

            if (viewerManager != null) {
                viewerManager.cleanUpOnClose();
            }

            if (layoutManager != null) {
                layoutManager.cleanUpOnClose();
            }

            transcriptionForThisFrame = null;
            viewerManager = null;
            layoutManager = null;
        }

        setJMenuBar(null);
        menuBar = null;
        dispose();
        System.runFinalization();
        System.gc();
    }

    /**
     * Adds an action to a menu. The menu is specified by an id, the position in
     * the menu can be specified by an index. The action is supposed to handle its
     * own events.
     *
     * @param action the action to add to a Menu (inside a JMenuItem)
     * @param menuId the identifier of the menu, a constant from #FrameConstants
     * @param index the index to insert the action, -1 means to add at the end
     */
    public void addActionToMenu(Action action, int menuId, int index) {
        if (action == null) {
            return;
        }

        Object mm = getMenuById(menuId);

        if (mm instanceof JMenu) {
            JMenu menu = (JMenu) mm;

            if (menu == menuWindow) {
                JRadioButtonMenuItem item = new JRadioButtonMenuItem(action);
                menu.add(item, index);
                windowsGroup.add(item);
            } else {
                JMenuItem item = new JMenuItem(action);
                menu.add(item, index);
            }
        }
    }

    /**
     * Looks for an Action with the specified id (== the LONG_DESCRIPTION) under
     * the menu identified by the menu id. If it is found it is removed from the menu
     * and returned.
     * @param actionId the id of the Action, currently stored in the LONG_DESCRIPTION value
     * @param menuId the identifier of the menu, a constant from #FrameConstants
     * @return the action that has been removed from the menu
     */
    public Action removeActionFromMenu(String actionId, int menuId) {
        if (actionId == null) {
            return null;
        }

        Object mm = getMenuById(menuId);

        if (mm instanceof JMenu) {
            JMenu menu = (JMenu) mm;
            JMenuItem item;
            Action ac;
            Object mi;

            for (int i = 0; i < menu.getMenuComponentCount(); i++) {
                mi = menu.getMenuComponent(i);

                if (mi instanceof JMenuItem) {
                    item = (JMenuItem) mi;
                    ac = item.getAction();

                    if (ac != null) {
                        if (actionId.equals(ac.getValue(Action.LONG_DESCRIPTION))) {
                            menu.remove(item);

                            if (menu == menuWindow) {
                                windowsGroup.remove(item);
                            }

                            return ac;
                        }
                    }
                }
            }
        }

        return null;
    }

    /**
     * Sets the action / menuitem identified by actionId selected.
     * @param actionId the id of the action
     * @param menuId the id of the menu, a constant from #FrameConstants
     */
    public void setMenuSelected(String actionId, int menuId) {
        if (actionId == null) {
            return;
        }

        Object mm = getMenuById(menuId);

        if (mm instanceof JMenu) {
            JMenu menu = (JMenu) mm;
            JRadioButtonMenuItem item;
            Action ac;
            Object mi;

            for (int i = 0; i < menu.getMenuComponentCount(); i++) {
                mi = menu.getMenuComponent(i);

                if (mi instanceof JRadioButtonMenuItem) {
                    item = (JRadioButtonMenuItem) mi;
                    ac = item.getAction();

                    if (ac != null) {
                        if (actionId.equals(ac.getValue(Action.LONG_DESCRIPTION))) {
                            item.setSelected(true);
                        }
                    }
                }
            }
        }
    }

    /**
     * Enables or disables a menu or menuitem, identified by meniId.
     * Most menu items can also be enabled/disabled through their
     * ActionCommands.
     *
     * @param menuId the identifier of the menu, a constant defined
     * in FrameConstants
     * @param enabled the enabled state
     */
    public void setMenuEnabled(int menuId, boolean enabled) {
        JMenuItem menu = getMenuById(menuId);

        if (menu != null) {
            menu.setEnabled(enabled);
        }
    }

    /**
     * Enables or disables all commands that are not in a menu but are instead
     * added to the menubar's action map (and are accessible through an accelerator
     * key). Disabling is performed by removing the actions from the action map, so
     * their original enabled state is not changed. Enabling is done by re-adding
     * the actions.
     *
     * @param enable
     */
    public void enableCommands(boolean enable) {
        ActionMap menuMap = menuBar.getActionMap();

        if (menuMap != null) {
            Iterator keyIt = registeredActions.keySet().iterator();
            Object key;

            while (keyIt.hasNext()) {
                key = keyIt.next();
                ((Action) registeredActions.get(key)).setEnabled(enable);
            }

            /*
            if (!enable) {
                menuMap.clear();
            } else {
                Iterator keyIt = registeredActions.keySet().iterator();
                Object key;
                while (keyIt.hasNext()) {
                    key = keyIt.next();
                    menuMap.put(key, (Action) registeredActions.get(key));
                }
            }
            */
        }
    }

    /**
     * Returns the menu item identified by one of the menu constants in
     * FrameConstants. This could be extended eventually to give access to
     * any menu (item).<br>
     * Note: the mapping from frame constant to menu or menuitem might need
     * to be reconsidered...
     *
     * @param id the id of the menu, as defined in FrameConstants
     * @return the corresponding menu or menuitem
     */
    private JMenuItem getMenuById(int id) {
        switch (id) {
        case FrameConstants.FILE:
            return menuFile;

        case FrameConstants.EDIT:
            return menuEdit;

        case FrameConstants.ANNOTATION:
            return menuAnnotation;

        case FrameConstants.TIER:
            return menuTier;

        case FrameConstants.TYPE:
            return menuType;

        case FrameConstants.SEARCH:
            return menuSearch;

        case FrameConstants.VIEW:
            return menuView;

        case FrameConstants.OPTION:
            return menuOptions;

        case FrameConstants.WINDOW:
            return menuWindow;

        case FrameConstants.HELP:
            return menuHelp;

        case FrameConstants.RECENT:
            return menuRecentFiles;

        case FrameConstants.EXPORT:
            return menuExport;

        case FrameConstants.IMPORT:
            return menuImport;

        case FrameConstants.LANG:
            return menuAppLanguage;

        case FrameConstants.PROPAGATION:
            return menuChangeTimePropMode;

        case FrameConstants.FRAME_LENGTH:
            return menuFrameLength;

        case FrameConstants.ANNOTATION_MODE:
            return menuItemAnnoMode;

        case FrameConstants.SYNC_MODE:
            return menuItemSyncMode;

        case FrameConstants.KIOSK_MODE:
            return menuItemKioskMode;

        case FrameConstants.PLAY_AROUND_SEL:
            return menuItemPlayAround;

        case FrameConstants.RATE_VOL_TOGGLE:
            return menuItemRateVol;
        }

        return null;
    }

    /**
     * DOCUMENT ME!
     *
     * @param arg DOCUMENT ME!
     */
    public static void main(String[] arg) {
        // supress floppy message from web start
        System.out.println("Java version: " +
            System.getProperty("java.version"));
        System.out.println("Runtime version: " +
            System.getProperty("java.runtime.version"));
        mpi.eudico.client.annotator.util.SystemInstallationSecurity.Instance()
                                                                   .go();

        //use java look and feel, so everything (multiple file chooser) looks the same
        //on windows and mac 
        /*
           try {
               UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
           } catch (Exception ex) {
               ex.printStackTrace();
           }
         */
        FrameManager.getInstance().setExitAllowed(true);

        //		String p2p = System.getProperty("P2P");
        //		if ((p2p != null) && (System.getProperty("P2P").equals("true"))) {
        //			mpi.eudico.p2p.ElanP2P.initP2P();
        //		}
        // make sure the directory for Elan data exists, for the time being the dir only contains global preferences
        File dataDir = new File(Constants.ELAN_DATA_DIR);

        if (!dataDir.exists()) {
            dataDir.mkdir();
        }

        // temporary, clean up old crap
        File oldCrap = new File(Constants.STRPROPERTIESFILE);
        oldCrap.delete();
        oldCrap = new File(Constants.USERHOME + Constants.FILESEPARATOR +
                ".elan.pfs");
        oldCrap.delete();

        if ((arg != null) && (arg.length > 0) && (arg[0].length() != 0)) {
            //new ElanFrame2(arg[0]);
            FrameManager.getInstance().createFrame(arg[0]);
        } else {
            //new ElanFrame2();
            FrameManager.getInstance().createEmptyFrame();
        }

        // preliminary external launcher 
        mpi.eudico.client.annotator.integration.ExternalLauncher.start();
    }

    /**
     * Switches between "Metal" and the Mac native Look and Feel. "Metal" on
     * MacOS isn't always behaving well; menus and popup menus  often are not
     * updated correctly etc.
     */
    private void switchMacLF() {
        if (menuMacNativeLF != null) {
            if (menuMacNativeLF.getState()) {
                try {
                    UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
                    SwingUtilities.updateComponentTreeUI(this);
                    Preferences.set("UseMacLF", new Boolean(true), null);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            } else {
                try {
                    UIManager.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
                    SwingUtilities.updateComponentTreeUI(this);
                    Preferences.set("UseMacLF", new Boolean(false), null);
                } catch (Exception ex) {
                    ex.printStackTrace();
                }
            }
        }
    }

    /**
     * Mac OS X specific handling of the main (screen) menu Quit application
     * event. Implementation of MacApplicationListener.
     */
    public void macHandleQuit() {
        FrameManager.getInstance().exit();
    }

    /**
     * Mac OS X specific handling of the main (screen) menu About application
     * event. Implementation of MacApplicationListener.
     */
    public void macHandleAbout() {
        if (menuActions != null) {
            MenuAction ma = null;

            for (int i = 0; i < menuActions.size(); i++) {
                ma = (MenuAction) menuActions.get(i);

                if (ma instanceof AboutMA) {
                    ma.actionPerformed(null);

                    break;
                }
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @throws Throwable DOCUMENT ME!
     */
    public void finalize() throws Throwable {
        System.out.println("Finalize ELAN window...");
        super.finalize();
    }

    /**
     * Listener for ElanFrame WindowEvents
     */
    private class ElanFrameWindowListener extends WindowAdapter {
        // triggered when the window is closed to quit the application
        // handle warnings and save operations in doExit();
        // EXIT WIL BE REPLACED BY CLOSE
        public void windowClosing(WindowEvent e) {
            //doExit();
            if (transcriptionForThisFrame != null) {
                // do nothing if this is an empty frame because a new
                // empty frame would be created
                checkSaveAndClose();
            } else {
                // 07-2007 changed behaviour: with one empty window, exit the 
                // application when the close button is pressed
                doClose(true);
            }
        }

        /**
         * Notifies the FrameManager that this frame has been activated.
         * Menus will be updated.
         */
        public void windowActivated(WindowEvent e) {
            FrameManager.getInstance().frameActivated(ElanFrame2.this);
        }
    }

    /*
       private class InitThread extends Thread {
           final String path;
           InitThread(String eafPath) {
               path = eafPath;
           }
           public void run() {
               final IndeterminateProgressMonitor monitor = new IndeterminateProgressMonitor(
                   ElanFrame2.this,
                   true,
                   ElanLocale.getString("Frame.ElanFrame.Progress.Open") + path,
                   false,
                   null);
               new Thread(new Runnable(){
                   public void run() {
                       monitor.show();
                   }
               }).start();
               //monitor.show();
               ElanFrame2.this.initElan();
               monitor.close();
           }
       }
     */
}
