/*
 * File:     ElanLocale.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.Vector;


/**
 * The class that handles everything with locales for Elan.
 */
public class ElanLocale {
    //private static Vector listeners = new Vector();
    private static Locale locale;
    private static ResourceBundle resourcebundle;
    private static HashMap listenerGroups = new HashMap();

    /** constant for dutch */
    public static final Locale DUTCH = new Locale("nl", "NL");

    /** constant for english, the default language */
    public static final Locale ENGLISH = new Locale("", "");

    /** constant for catalan */
    public static final Locale CATALAN = new Locale("ca");

    /** constant for spanish */
    public static final Locale SPANISH = new Locale("es", "ES");

    /** constant for swedish */
    public static final Locale SWEDISH = new Locale("sv", "SE");

    /** constant for german */
    public static final Locale GERMAN = new Locale("de", "DE");

    /** constant for portuguese */
    public static final Locale PORTUGUESE = new Locale("pt");

    /** constant for french */
    public static final Locale FRENCH = new Locale("fr");

    /**
     * Constructor
     */
    ElanLocale() {
        locale = Locale.getDefault();
        resourcebundle = ResourceBundle.getBundle("mpi.eudico.client.annotator.resources.ElanLanguage",
                locale);
    }

    /**
     * Gets the current locale
     *
     * @return The current locale
     */
    public static Locale getLocale() {
        return locale;
    }

    /**
     * Sets the current locale
     *
     * @param locale_in The new locale
     */
    public static void setLocale(Locale locale_in) {
        if ((locale != null) && locale.equals(locale_in)) {
            return;
        }

        //if (locale.getCountry().equals(locale_in.getCountry())) {
        //	return;
        //}
        locale = locale_in;
        resourcebundle = ResourceBundle.getBundle("mpi.eudico.client.annotator.resources.ElanLanguage",
                locale);
        notifyListeners();

        try {
            mpi.search.SearchLocale.setLocale(locale_in);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Gets the string in the right language from the right resource file
     *
     * @param str The string which has to be mapped to the right language
     *
     * @return The string in the right language
     */
    public static String getString(String str) {
        if (locale == null) {
            locale = Locale.getDefault();
            resourcebundle = ResourceBundle.getBundle("mpi.eudico.client.annotator.resources.ElanLanguage",
                    locale);
        }

        try {
            return resourcebundle.getString(str);
        } catch (Exception ex) {
            return "";
        }
    }

    /**
     * Adds an Elan Locale listener.
     *
     * @param listener A new Elan Locale listener
     */
    public static void addElanLocaleListener(Object key,
        ElanLocaleListener listener) {
        if (listenerGroups.containsKey(key)) {
            ((ArrayList) listenerGroups.get(key)).add(listener);

            listener.updateLocale();
        } else {
            ArrayList list = new ArrayList();
            list.add(listener);

            listenerGroups.put(key, list);
            listener.updateLocale();
        }

        /*
        if (!listeners.contains(listener)) {
            listeners.add(listener);

            // make sure the listener is up to date
            listener.updateLocale();
        }
        */
    }

    /**
     * Removes an Elan Locale listener.
     *
     * @param listener The listener which has to be removed
     */
    public static void removeElanLocaleListener(ElanLocaleListener listener) {
        //listeners.remove(listener);
        Iterator groupIt = listenerGroups.keySet().iterator();
        ArrayList listeners;

        while (groupIt.hasNext()) {
            listeners = (ArrayList) listenerGroups.get(groupIt.next());

            if (listeners.remove(listener)) {
                break;
            }
        }
    }

    /**
     * Removes an Elan Locale listener group.
     *
     * @param key The key of the group which has to be removed
     */
    public static void removeElanLocaleListener(Object key) {
        //listeners.remove(listener);
        listenerGroups.remove(key);
    }

    /**
     * Notifies all listeners if the locale has been changed.
     */
    private static void notifyListeners() {
        //for (int i = 0; i < listeners.size(); i++) {
        //	((ElanLocaleListener) listeners.elementAt(i)).updateLocale();
        //}
        Iterator groupIt = listenerGroups.keySet().iterator();
        ArrayList listeners;

        while (groupIt.hasNext()) {
            listeners = (ArrayList) listenerGroups.get(groupIt.next());

            for (int i = 0; i < listeners.size(); i++) {
                ((ElanLocaleListener) listeners.get(i)).updateLocale();
            }
        }
    }

    /**
     * Returns the resource bundle for use in (generic) classes that do not
     * directly access ElanLocale.
     *
     * @return the resource bundle
     */
    public static ResourceBundle getResourceBundle() {
        return resourcebundle;
    }
}
