/*
 * File:     AnnotationAfterCommand.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.AnnotationDataRecord;

import mpi.eudico.server.corpora.clom.Annotation;
import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.AlignableAnnotation;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;


/**
 * A command for creating a new annotation after a given annotation.
 *
 * @author Han Sloetjes
 */
public class AnnotationAfterCommand implements UndoableCommand {
    private String commandName;
    private TierImpl tier;
    private Transcription transcription;
    private AnnotationDataRecord activeAnnRecord;
    private long aaMiddle;
    private long newAnnMiddle;

    /**
     * Creates a new AnnotationAfterCommand instance
     *
     * @param name the name of the command
     */
    public AnnotationAfterCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action.
     */
    public void undo() {
        if (tier != null) {
            Annotation delAnn = tier.getAnnotationAtTime(newAnnMiddle);

            if (delAnn != null) {
                tier.removeAnnotation(delAnn);
            }

            if (tier.isTimeAlignable()) {
                AlignableAnnotation activeAnn = (AlignableAnnotation) (tier.getAnnotationAtTime(aaMiddle));

                if ((activeAnn != null) &&
                        ((activeAnn.getBegin().isTimeAligned() != activeAnnRecord.isBeginTimeAligned()) ||
                        (activeAnn.getEnd().isTimeAligned() != activeAnnRecord.isEndTimeAligned()))) {
                    int curPropMode = 0;

                    curPropMode = transcription.getTimeChangePropagationMode();

                    if (curPropMode != Transcription.NORMAL) {
                        transcription.setTimeChangePropagationMode(Transcription.NORMAL);
                    }

                    activeAnn.updateTimeInterval(activeAnnRecord.getBeginTime(),
                        activeAnnRecord.getEndTime());

                    // restore the time propagation mode
                    transcription.setTimeChangePropagationMode(curPropMode);
                }
            }
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if (tier != null) {
            Annotation afterAnn = tier.getAnnotationAtTime(aaMiddle);

            if (afterAnn != null) {
                tier.createAnnotationAfter(afterAnn);
            }
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the TierImpl
     * @param arguments the arguments:  <ul><li>arg[0] = the active annotation
     *        (Annotation)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        tier = (TierImpl) receiver;

        Annotation activeAnn = (Annotation) arguments[0];
        activeAnnRecord = new AnnotationDataRecord(activeAnn);
        aaMiddle = (activeAnnRecord.getBeginTime() +
            activeAnnRecord.getEndTime()) / 2;

        transcription = (Transcription) tier.getParent();

        Annotation newAnnotation = tier.createAnnotationAfter(activeAnn);

        if (newAnnotation != null) {
            newAnnMiddle = (newAnnotation.getBeginTimeBoundary() +
                newAnnotation.getEndTimeBoundary()) / 2;
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
