/*
 * File:     ChangeCVCommand.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import mpi.util.ControlledVocabulary;


/**
 * A Command to change a Controlled Vocabulary (name and or description).
 *
 * @author Han Sloetjes
 */
public class ChangeCVCommand implements UndoableCommand {
    private String commandName;

    // receiver; the transcription handles the change of the name or description 
    // of a CV
    private TranscriptionImpl transcription;

    // store the arguments for undo /redo
    private String oldCVName;
    private String oldCVDesc;
    private String cvName;
    private String description;
    private ControlledVocabulary conVoc;

    /**
     * Creates a new ChangeCVCommand instance
     *
     * @param name the name of the command
     */
    public ChangeCVCommand(String name) {
        commandName = name;
    }

    /**
     * The undo action. Restores the old values of the CV.
     */
    public void undo() {
        if ((transcription != null) && (conVoc != null)) {
            transcription.changeControlledVocabulary(conVoc, oldCVName,
                oldCVDesc);
        }
    }

    /**
     * The redo action.
     */
    public void redo() {
        if ((transcription != null) && (conVoc != null)) {
            transcription.changeControlledVocabulary(conVoc, cvName, description);
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the Transcription
     * @param arguments the arguments: <ul><li>arg[0] = the old name of the
     *        Controlled Vocabulary  (String)</li> <li>arg[1] = the old
     *        description of the CV (String)</li> <li>arg[2] = the new name of
     *        the Controlled Vocabulary  (String)</li> <li> arg[3] =  the new
     *        description of the CV (String)</li> </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (TranscriptionImpl) receiver;
        oldCVName = (String) arguments[0];

        if (arguments[1] != null) {
            oldCVDesc = (String) arguments[1];
        }

        cvName = (String) arguments[2];

        if (arguments[3] != null) {
            description = (String) arguments[3];
        }

        conVoc = transcription.getControlledVocabulary(oldCVName);

        if (conVoc != null) {
            transcription.changeControlledVocabulary(conVoc, cvName, description);
        }
    }

    /**
     * Returns the name of the command.
     *
     * @return the name of the command
     */
    public String getName() {
        return commandName;
    }
}
