/*
 * File:     DuplicateAnnotationCommand.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clom.Annotation;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;

import java.util.Vector;


/**
 * An undoable command that builds on the new annotation command.
 * It takes the active annotation as input, creates a new annotation and sets the value for the new annotation
 * to be the same as the input annotation. This is then one undoable command in the undo/redo list.
 */
public class DuplicateAnnotationCommand extends NewAnnotationCommand {
    private String value;

    /**
     * Creates a new DuplicateAnnotationCommand instance
     *
     * @param name the name of the command
     */
    public DuplicateAnnotationCommand(String name) {
        super(name);
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct.
     *
     * @param receiver the ViewerManager
     * @param arguments the arguments:  <ul><li>arg[0] = the tier
     *        (TierImpl)</li> <li>arg[1] = the annotation to duplicate
     *        (Annotation)</li>  </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        ViewerManager2 vm = (ViewerManager2) receiver;
        TierImpl t = (TierImpl) arguments[0];
        Annotation ann = (Annotation) arguments[1];
        value = ann.getValue();

        // adjust bt and et if necesary
        long[] times = adjustTimes(t, ann.getBeginTimeBoundary(),
                ann.getEndTimeBoundary());

        if (times != null) {
            super.execute(t,
                new Object[] { new Long(times[0]), new Long(times[1]) });

            // set value
            if (newAnnotation != null) {
                newAnnotation.setValue(value);
            }

            Command c = ELANCommandFactory.createCommand(vm.getTranscription(),
                    ELANCommandFactory.ACTIVE_ANNOTATION);
            c.execute(vm, new Object[] { ann });
        }
    }

    /**
     * @see mpi.eudico.client.annotator.commands.UndoableCommand#redo()
     */
    public void redo() {
        super.redo();

        if (newAnnotation != null) {
            newAnnotation.setValue(value);
        }
    }

    /**
     * @see mpi.eudico.client.annotator.commands.UndoableCommand#undo()
     */
    public void undo() {
        super.undo();
    }

    /**
     * Adjust the begin and end time for the new annotation based on tier type and parent tier
     * and parent annotation.
     *
     * @param tier the destination tier
     * @param begin the begin time of the original annotation
     * @param end the end time of the original annotation
     * @return the adjusted begin and end time in an array
     */
    private long[] adjustTimes(TierImpl tier, long begin, long end) {
        long[] result = new long[] { begin, end };

        Constraint c = tier.getLinguisticType().getConstraints();

        if (c == null) {
            return result;
        }

        if ((c.getStereoType() == Constraint.SYMBOLIC_ASSOCIATION) ||
                (c.getStereoType() == Constraint.SYMBOLIC_SUBDIVISION)) {
            TierImpl par = (TierImpl) tier.getParentTier();
            long mid = (begin + end) / 2;

            if ((tier.getAnnotationAtTime(mid) == null) &&
                    (par.getAnnotationAtTime(mid) != null)) {
                result[0] = mid;
                result[1] = mid;
            } else if ((tier.getAnnotationAtTime(begin) == null) &&
                    (par.getAnnotationAtTime(begin) != null)) {
                result[1] = begin;
            } else if ((tier.getAnnotationAtTime(end) == null) &&
                    (par.getAnnotationAtTime(end) != null)) {
                result[0] = end;
            } else {
                // find any other overlapping annotation?
                return null;
            }
        } else if ((c.getStereoType() == Constraint.INCLUDED_IN) ||
                (c.getStereoType() == Constraint.TIME_SUBDIVISION)) {
            TierImpl par = (TierImpl) tier.getParentTier();
            Annotation ann1 = par.getAnnotationAtTime(begin);
            Annotation ann2 = par.getAnnotationAtTime(end);

            if ((ann1 != null) && (ann2 != null) && (ann1 != ann2)) {
                result[1] = ann1.getEndTimeBoundary();
            } else if ((ann1 != null) && (ann2 == null)) {
                result[1] = ann1.getEndTimeBoundary();
            } else if ((ann1 == null) && (ann2 != null)) {
                result[0] = ann2.getBeginTimeBoundary();
            } else {
                Vector v = par.getOverlappingAnnotations(begin, end);

                if (v.size() == 0) {
                    return null;
                } else {
                    ann1 = (Annotation) v.get(0); // take the first one
                    result[0] = ann1.getBeginTimeBoundary();
                    result[1] = ann1.getEndTimeBoundary();
                }
            }
        }

        return result;
    }
}
