/*
 * File:     ExportPrefsCommand.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.server.corpora.clom.Transcription;

import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.filechooser.FileFilter;


/**
 * Shows a save dialog and passes the selected filepath to the Preferences
 * export method.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public class ExportPrefsCommand implements Command {
    private String commandName;
    private Transcription transcription;

    /**
     * Creates a new ExportPrefsCommand instance
     *
     * @param name name of the command
     */
    public ExportPrefsCommand(String name) {
        commandName = name;
    }

    /**
     * Shows the save dialog and exports the Preferences.
     *
     * @param receiver the Transcription/document of which to save the
     *        preferences
     * @param arguments null
     */
    public void execute(Object receiver, Object[] arguments) {
        transcription = (Transcription) receiver;

        if (transcription == null) {
            return;
        }

        String filePath = promptForExportFile();

        if (filePath == null) {
            return;
        }

        Preferences.exportPreferences(transcription, filePath);
    }

    /**
     * Prompts the user to specify a location and name for the prefs file.
     *
     * @return the file path
     */
    private String promptForExportFile() {
        String prefDir = (String) Preferences.get("LastUsedExportDir", null);

        if (prefDir == null) {
            prefDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser();

        chooser.setApproveButtonText(ElanLocale.getString("Button.Save"));
        chooser.setCurrentDirectory(new File(prefDir));
        chooser.setDialogTitle(ElanLocale.getString("ExportDialog.ExportToFile"));

        File prefFile = null;
        FileFilter filter = ElanFileFilter.createFileFilter(ElanFileFilter.ELAN_PREFS_TYPE);
        chooser.setAcceptAllFileFilterUsed(false);
        chooser.setFileFilter(filter);

        if (chooser.showSaveDialog(ELANCommandFactory.getRootFrame(
                        transcription)) == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedExportDir", curDir.getAbsolutePath(),
                    null);
            }

            prefFile = chooser.getSelectedFile();

            if (prefFile != null) {
                String name = prefFile.getAbsolutePath();
                String lowerPathName = name.toLowerCase();
                String[] extensions = FileExtension.ELAN_XML_PREFS_EXT;

                boolean validExt = false;

                for (int i = 0; i < extensions.length; i++) {
                    if (lowerPathName.endsWith("." +
                                extensions[i].toLowerCase())) {
                        validExt = true;

                        break;
                    }
                }

                if (!validExt) {
                    name += ("." + extensions[0]);
                    prefFile = new File(name);
                }

                if (prefFile.exists()) {
                    int answer = JOptionPane.showConfirmDialog(ELANCommandFactory.getRootFrame(
                                transcription),
                            ElanLocale.getString("Message.Overwrite"),
                            ElanLocale.getString("SaveDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer == JOptionPane.NO_OPTION) {
                        return promptForExportFile();
                    }
                }

                return name;
            }
        }

        return null;
    }

    /**
     * Returns the name.
     *
     * @return the name!
     */
    public String getName() {
        return commandName;
    }
}
