/*
 * File:     KeyCreateAnnotationCA.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ViewerManager2;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;
import javax.swing.KeyStroke;


/**
 * A CommandAction to create an annotation by pressing a key combination
 * alternatively at the  begin time and the end time of the desired new
 * annotation.
 */
public class KeyCreateAnnotationCA extends CommandAction {
    private TierImpl receiver;
    private long firstTime = -1;
    private long secTime = -1;

    /**
     * Constructor.
     *
     * @param vm the viewer manager
     */
    public KeyCreateAnnotationCA(ViewerManager2 vm) {
        super(vm, ELANCommandFactory.KEY_CREATE_ANNOTATION);

        putValue(Action.ACCELERATOR_KEY,
            KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, ActionEvent.SHIFT_MASK));
        putValue(SHORT_DESCRIPTION,
            ElanLocale.getString(ELANCommandFactory.KEY_CREATE_ANNOTATION +
                "ToolTip"));
    }

    /**
     * @see mpi.eudico.client.annotator.commands.CommandAction#newCommand()
     */
    protected void newCommand() {
        command = null;

        if (receiver != null) {
            command = ELANCommandFactory.createCommand(vm.getTranscription(),
                    ELANCommandFactory.NEW_ANNOTATION);
        }
    }

    /**
     * Returns the current interval, begin and end time and resets the internal
     * first  key stroke and second key stroke values.
     *
     * @return an Object array containing 2 Longs, begin and end time
     */
    protected Object[] getArguments() {
        if (firstTime < secTime) {
            Long b = new Long(firstTime);
            Long e = new Long(secTime);

            //reset
            firstTime = -1;
            secTime = -1;

            return new Object[] { b, e };
        } else {
            Long e = new Long(firstTime);
            Long b = new Long(secTime);

            //reset
            firstTime = -1;
            secTime = -1;

            return new Object[] { b, e };
        }
    }

    /**
     * Returns the tier to which a new annotation should be added.
     *
     * @see mpi.eudico.client.annotator.commands.CommandAction#getReceiver()
     */
    protected Object getReceiver() {
        return receiver;
    }

    /**
     * Checks whether it is the first or the second key stroke, if the active
     * tier is time alignable, if at the position of the interval an
     * annotation can be created etc.
     *
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent event) {
        // check if this is the first or second key stroke
        if (firstTime == -1) {
            firstTime = vm.getMasterMediaPlayer().getMediaTime();
        } else {
            long st = vm.getMasterMediaPlayer().getMediaTime();

            if (st == firstTime) {
                return; // do nothing
            }

            if ((vm.getMultiTierControlPanel() != null) &&
                    (vm.getMultiTierControlPanel().getActiveTier() != null)) {
                TierImpl ti = (TierImpl) vm.getMultiTierControlPanel()
                                           .getActiveTier();

                if (ti.isTimeAlignable()) {
                    if (!ti.hasParentTier()) {
                        secTime = st;
                        receiver = ti;
                        super.actionPerformed(event);

                        return;
                    } else {
                        if ((((TierImpl) ti.getParentTier()).getAnnotationAtTime(
                                    firstTime) != null) ||
                                (((TierImpl) ti.getParentTier()).getAnnotationAtTime(
                                    st) != null)) {
                            // the interval will be forced within the parent's elsewhere
                            secTime = st;
                            receiver = ti;
                            super.actionPerformed(event);

                            return;
                        }
                    }
                }
            }

            firstTime = st; // treat this time as the first time
        }

        //super.actionPerformed(event);
    }
}
