/*
 * File:     PasteAnnotationCommand.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.util.AnnotationDataFlavor;
import mpi.eudico.client.annotator.util.AnnotationDataRecord;
import mpi.eudico.client.annotator.util.AnnotationTreeDataFlavor;
import mpi.eudico.client.annotator.util.ClientLogger;

import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.Toolkit;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;

import java.io.IOException;

import javax.swing.tree.DefaultMutableTreeNode;


/**
 * A Command to paste an annotation (i.e. a transferable AnnotationDataRecord) from the System's
 * Clipboard.
 *
 * Note June 2006 transferal of application/x-java-serialized-object objects between jvm's
 * doesn't seem to work on Mac OS X. The Transferable sun.awt.datatransfer.ClipboardTransferable@8b4cb8
 * doesn't support any flavor with human presentable name "application/x-java-serialized-object"
 */
public class PasteAnnotationCommand extends NewAnnotationCommand
    implements ClientLogger {
    private AnnotationDataRecord record;
    private String destTierName;
    private Long newBeginTime;

    /**
     * Creates a new PasteAnnotationCommand instance
     *
     * @param name the name of the command
     */
    public PasteAnnotationCommand(String name) {
        super(name);
    }

    /**
     * @see mpi.eudico.client.annotator.commands.UndoableCommand#undo()
     */
    public void undo() {
        super.undo();
    }

    /**
     * @see mpi.eudico.client.annotator.commands.UndoableCommand#redo()
     */
    public void redo() {
        super.redo();

        if ((newAnnotation != null) && (record != null)) {
            newAnnotation.setValue(record.getValue());
        }
    }

    /**
     * <b>Note: </b>it is assumed the types and order of the arguments are
     * correct. The arguments parameter can be null, of length 1 or of length 2<br>
     * If the arguments parameter is null the following order is used to decide on which tier the
     * annotation has te be pasted:<br>
     * - the tier of the same name as in the copied annotation record, if present in the destination transcription<br>
     * - the current active tier
     *
     * @param receiver a transcription
     * @param arguments the arguments:  <ul><li>null</li></ul>
     * or
     * <ul><li>arg[0] = destination tier name if the annotation should be pasted to a tier with another
     * name (String)</li> <li>arg[1] = destination begin time; if the annotation should be pasted at
     * another location (Long)</li>
     * </ul>
     */
    public void execute(Object receiver, Object[] arguments) {
        TranscriptionImpl tr = (TranscriptionImpl) receiver;

        if (canAccessSystemClipboard()) {
            Object contents = null;
            Transferable ta;

            // get Clipboard contents.
            ta = Toolkit.getDefaultToolkit().getSystemClipboard().getContents(null);

            try {
                contents = ta.getTransferData(AnnotationDataFlavor.getInstance());
            } catch (UnsupportedFlavorException nfe) {
                // check if there is an annotation tree on the clipboard 
                try {
                    contents = ta.getTransferData(AnnotationTreeDataFlavor.getInstance());
                } catch (Exception e) {
                    LOG.warning("Unsupported flavor on the clipboord: " +
                        nfe.getMessage());

                    return;
                }

                /* could use a String representation as alternative?
                try {
                    contents = ta.getTransferData(DataFlavor.stringFlavor);
                } catch (Exception e) {
                    LOG.warning("No text on clipbaard: " + e.getMessage());
                    return;
                }
                */

                //return;
            } catch (IOException ioe) {
                LOG.warning("I/O error: " + ioe.getMessage());

                return;
            }

            if (contents instanceof DefaultMutableTreeNode) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) contents;
                record = (AnnotationDataRecord) node.getUserObject();
            } else if (contents instanceof AnnotationDataRecord) {
                record = (AnnotationDataRecord) contents;
            } else {
                return;
            }

            destTierName = record.getTierName();

            //LOG.info("Record: " + record.toString() + " " + record.getBeginTime() +
            //        " " + record.getEndTime());
            if ((arguments != null) && (arguments.length > 0)) {
                if (arguments[0] != null) {
                    destTierName = (String) arguments[0];
                }

                if (arguments.length > 1) {
                    newBeginTime = (Long) arguments[1];
                }
            }

            TierImpl tier = null;

            if (destTierName != null) {
                tier = (TierImpl) tr.getTierWithId(destTierName);
            }

            // if the tier is not found try the active tier
            if (tier == null) {
                tier = (TierImpl) ELANCommandFactory.getViewerManager(tr)
                                                    .getMultiTierControlPanel()
                                                    .getActiveTier();

                if (tier != null) {
                    destTierName = tier.getName();
                }
            }

            if (tier == null) {
                LOG.warning("Cannot paste annotation: tier not found: " +
                    destTierName);

                return;
            }

            Long[] args = new Long[2];

            if (newBeginTime != null) {
                if (tier.isTimeAlignable()) {
                    args[0] = newBeginTime;
                    args[1] = new Long(newBeginTime.longValue() +
                            (record.getEndTime() - record.getBeginTime()));
                } else {
                    args[0] = newBeginTime;
                    args[1] = newBeginTime;
                }
            } else {
                if (tier.isTimeAlignable()) {
                    args[0] = new Long(record.getBeginTime());
                    args[1] = new Long(record.getEndTime());
                } else {
                    long mid = (record.getBeginTime() + record.getEndTime()) / 2;
                    args[0] = new Long(mid);
                    args[1] = args[0];
                }
            }

            super.execute(tier, args);
        }
    }

    /**
     * The creation of the new annotation in a separate method to allow overriding.
     */
    void newAnnotation() {
        super.newAnnotation();

        if (newAnnotation != null) {
            newAnnotation.setValue(record.getValue());
        }
    }

    /**
     * Performs a check on the accessibility of the system clipboard.
     *
     * @return true if the system clipboard is accessible, false otherwise
     */
    protected boolean canAccessSystemClipboard() {
        if (System.getSecurityManager() != null) {
            try {
                System.getSecurityManager().checkSystemClipboardAccess();

                return true;
            } catch (SecurityException se) {
                LOG.warning("Cannot access the clipboard");

                //se.printStackTrace();
                return false;
            }
        }

        return true;
    }
}
