/*
 * File:     PlaybackRateToggleCA.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.ViewerManager2;

import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;

import javax.swing.Action;
import javax.swing.KeyStroke;


/**
 * A CommandAction that stores and sets the toggle value for  the playback rate
 * shortcut.
 *
 * @author Han Sloetjes
 */
public class PlaybackRateToggleCA extends CommandAction {
    /** the default rate */
    private final float defaultRate = 1.0F;

    /** the initial toggle value */
    private float toggleValue = 0.5F;

    //start with the default rate
    private boolean isToggled = false;

    /**
     * Creates a new PlaybackRateToggleCA instance
     *
     * @param viewerManager the viewer manager
     */
    public PlaybackRateToggleCA(ViewerManager2 viewerManager) {
        super(viewerManager, ELANCommandFactory.PLAYBACK_RATE_TOGGLE);

        putValue(Action.ACCELERATOR_KEY,
            KeyStroke.getKeyStroke(KeyEvent.VK_R,
                Toolkit.getDefaultToolkit().getMenuShortcutKeyMask() +
                ActionEvent.ALT_MASK));
        putValue(SHORT_DESCRIPTION,
            ElanLocale.getString(ELANCommandFactory.PLAYBACK_RATE_TOGGLE +
                "ToolTip"));

        //putValue(Action.NAME, "");
    }

    /**
     * Creates a new PlaybackRateToggleCommand. Every time newCommand is
     * called the rate value should be toggled.
     */
    protected void newCommand() {
        isToggled = !isToggled;

        command = ELANCommandFactory.createCommand(vm.getTranscription(),
                ELANCommandFactory.PLAYBACK_RATE_TOGGLE);
    }

    /**
     * Returns the reciever, the viewermanager. The command should set the rate
     * of the master media player  and update the rate slider.
     *
     * @return the viewermanager
     */
    protected Object getReceiver() {
        return vm;
    }

    /**
     * Returns the new rate.
     *
     * @return the new rate
     */
    protected Object[] getArguments() {
        Object[] args = new Object[1];

        if (isToggled) {
            args[0] = new Float(toggleValue);
        } else {
            args[0] = new Float(defaultRate);
        }

        return args;
    }

    /**
     * Sets the toggle value. The value should be between
     * 0 and 200, inclusive.
     *
     * @param value the toggle value
     */
    public void setToggleValue(float value) {
        if ((value >= 0F) && (value <= 2.0F)) {
            toggleValue = value;
        }
    }

    /**
     * Returns the toggle value.
     *
     * @return the toggle value.
     */
    public float getToggleValue() {
        return toggleValue;
    }
}
