/*
 * File:     AbstractProcessMultiMA.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands.global;

import mpi.eudico.client.annotator.ElanFrame2;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.gui.MultiFileChooser;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import java.io.File;

import java.util.ArrayList;

import javax.swing.JFileChooser;
import javax.swing.JFrame;


/**
 * Base class for processing multiple eaf files. This part deals with
 * selecting multiple files and directories and creating a list of unique,
 * existing files.
 *
 * @author Han Sloetjes
 * @version 1.0
 */
public abstract class AbstractProcessMultiMA extends FrameMenuAction {
    /**
     * Creates a new AbstractProcessMultiMA instance
     *
     * @param name name of the action
     * @param frame the containing frame
     */
    public AbstractProcessMultiMA(String name, ElanFrame2 frame) {
        super(name, frame);
    }

    /**
     * Shows a multiple file chooser to select multiple eaf files and or
     * folders.
     *
     * @param parent the parent frame
     * @param title the title for the dialog
     *
     * @return a list of File objects (files and folders)
     */
    protected ArrayList getMultipleFiles(JFrame parent, String title) {
        ArrayList files = new ArrayList();

        MultiFileChooser chooser = new MultiFileChooser();
        chooser.setDialogTitle(title);
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.EAF_TYPE));
        chooser.setFileSelectionMode(JFileChooser.FILES_AND_DIRECTORIES);

        String dirPath = (String) Preferences.get("LastUsedEAFDir", null);

        if (dirPath == null) {
            // user.dir is probably a better choice than home.dir?
            dirPath = System.getProperty("user.dir");
        }

        chooser.setCurrentDirectory(new File(dirPath));

        // let the user choose
        int option = chooser.showDialog(parent, null);

        if (option == JFileChooser.APPROVE_OPTION) {
            String lastDirPath = chooser.getCurrentDirectory().getAbsolutePath();
            Preferences.set("LastUsedEAFDir", lastDirPath, null);

            Object[] names = chooser.getFiles();

            for (int i = 0; i < names.length; i++) {
                String name = "" + names[i];
                File f = new File(name);

                if (f.isFile() && f.canRead()) {
                    files.add(f);
                } else if (f.isDirectory() && f.canRead()) {
                    addFiles(f, files);
                }
            }
        }

        return files;
    }

    /**
     * Scans the folders for eaf files and adds them to files list,
     * recursively.
     *
     * @param dir the  or folder
     * @param files the list to add the files to
     */
    protected void addFiles(File dir, ArrayList files) {
        if ((dir == null) && (files == null)) {
            return;
        }

        File[] allSubs = dir.listFiles();

        for (int i = 0; i < allSubs.length; i++) {
            if (allSubs[i].isDirectory() && allSubs[i].canRead()) {
                addFiles(allSubs[i], files);
            } else {
                if (allSubs[i].canRead()) {
                    if (allSubs[i].getName().toLowerCase().endsWith(FileExtension.EAF_EXT[0])) {
                        // test if the file is already there??
                        files.add(allSubs[i]);
                    }
                }
            }
        }
    }
}
