/*
 * File:     ImportCHATMA.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.commands.global;

import mpi.eudico.client.annotator.ElanFrame2;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.FrameManager;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.util.ElanFileFilter;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;

import java.awt.event.ActionEvent;

import java.io.File;

import javax.swing.JFileChooser;
import javax.swing.JOptionPane;


/**
 * Action that starts an Import CHAT sequence.
 *
 * @author Han Sloetjes, MPI
 */
public class ImportCHATMA extends FrameMenuAction {
    /**
     * Creates a new ImportCHATMA instance.
     *
     * @param name the name of the action (command)
     * @param frame the associated frame
     */
    public ImportCHATMA(String name, ElanFrame2 frame) {
        super(name, frame);
    }

    /**
     * Shows an import CHAT dialog and creates a new transcription.
     *
     * @param ae the action event
     */
    public void actionPerformed(ActionEvent ae) {
        JOptionPane.showMessageDialog(frame,
            ElanLocale.getString("ExportCHATDialog.Message.CLANutility"),
            "ELAN", JOptionPane.INFORMATION_MESSAGE);

        JFileChooser chooser = new JFileChooser();

        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.CHAT_TYPE));
        chooser.setDialogTitle(ElanLocale.getString(
                "Frame.ElanFrame.OpenDialog.Title"));

        String dirPath = (String) Preferences.get("LastUsedCHATDir", null);

        if (dirPath == null) {
            // user.dir is probably a better choice than home.dir?
            dirPath = System.getProperty("user.dir");
        }

        chooser.setCurrentDirectory(new File(dirPath));

        int returnVal = chooser.showOpenDialog(frame);

        if (returnVal == JFileChooser.APPROVE_OPTION) {
            String fullPath = chooser.getSelectedFile().getAbsolutePath();
            File fileTemp = new File(fullPath);

            //check if file exists and is a file
            if (!fileTemp.exists() || fileTemp.isDirectory()) {
                String strMessage = ElanLocale.getString("Menu.Dialog.Message1");
                strMessage += fullPath;
                strMessage += ElanLocale.getString("Menu.Dialog.Message2");

                String strError = ElanLocale.getString("Message.Error");
                JOptionPane.showMessageDialog(frame, strMessage, strError,
                    JOptionPane.ERROR_MESSAGE);

                return;
            }

            //check if file is a '.cha' file
            if (fileTemp.toString().toLowerCase().endsWith(".cha") == false) {
                String strMessage = ElanLocale.getString("Menu.Dialog.Message1");
                strMessage += fullPath;
                strMessage += ElanLocale.getString("Menu.Dialog.Message3");

                String strError = ElanLocale.getString("Message.Error");
                JOptionPane.showMessageDialog(frame, strMessage, strError,
                    JOptionPane.ERROR_MESSAGE);

                return;
            }

            //open the cha and get the Transcription from it
            try {
                // When config files are possible check if eaf or configuration file
                //           String path = chooser.getSelectedFile().getAbsolutePath();
                String path = fullPath;
                Preferences.set("LastUsedCHATDir",
                    (new File(path)).getParent(), null);

                // replace all backslashes by forward slashes
                path = path.replace('\\', '/');

                //long before = System.currentTimeMillis();
                Transcription transcription = new TranscriptionImpl(new File(
                            path).getAbsolutePath());

                //long after = System.currentTimeMillis();
                //System.out.println("open eaf took " + (after - before) + "ms");
                transcription.setChanged();

                int lastSlash = path.lastIndexOf('/');
                String chatPath = path.substring(0, lastSlash);
                boolean validMedia = true;

                if (frame != null) {
                    frame.checkMedia(transcription, chatPath);
                }

                if (!validMedia) {
                    // ask if no media session is ok, if not return
                    int answer = JOptionPane.showConfirmDialog(frame,
                            ElanLocale.getString(
                                "Frame.ElanFrame.IncompleteMediaQuestion"),
                            ElanLocale.getString(
                                "Frame.ElanFrame.IncompleteMediaAvailable"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer != JOptionPane.YES_OPTION) {
                        return;
                    }
                }

                FrameManager.getInstance().createFrame(transcription);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }
}
