/*
 * File:     AbstractBasicExportDialog.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.gui.ClosableDialog;

import mpi.eudico.client.annotator.util.ClientLogger;
import mpi.eudico.client.annotator.util.ElanFileFilter;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.util.gui.FileAndEncodingChooser;

import java.awt.Frame;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import java.io.File;
import java.io.IOException;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;
import javax.swing.filechooser.FileFilter;


/**
 * $Id: AbstractBasicExportDialog.java 10724 2007-10-29 13:52:34Z hasloe $
 *
 * @author $author$
 * @version $Revision$
 */
public abstract class AbstractBasicExportDialog extends ClosableDialog
    implements ActionListener, ClientLogger {
    /** key to store last used export directory in preferences file */
    public static final String LAST_USED_EXPORT_DIR = "LastUsedExportDir";

    /** insets between subcomponents */
    protected final Insets insets = new Insets(4, 6, 4, 6);

    /** header in window (top component) */
    protected final JLabel titleLabel = new JLabel();

    /** panel for start and close buttons (bottom component) */
    protected final JPanel buttonPanel = new JPanel();

    /** panel for export options (one of 'body' components) */
    protected final JPanel optionsPanel = new JPanel();

    /** store default as UTF-8 */
    protected final String defaultEncoding = "UTF-8";

    /** table model for tier table */
    protected final Transcription transcription;

    /** Character Encoding of export file */
    protected String encoding = defaultEncoding;

    /** close button */
    private final JButton closeButton = new JButton();

    /** start export button */
    private final JButton startButton = new JButton();

    /** minimal window height */
    private final int minimalHeight = 400;

    /** minimal window width */
    private final int minimalWidth = 550;

    /**
     * Creates a new AbstractBasicExportDialog object.
     *
     * @param parent DOCUMENT ME!
     * @param modal DOCUMENT ME!
     * @param transcription DOCUMENT ME!
     */
    public AbstractBasicExportDialog(Frame parent, boolean modal,
        Transcription transcription) {
        super(parent, modal);
        this.transcription = transcription;
    }

    /**
     * The action performed event handling.
     *
     * @param ae the action event
     */
    public void actionPerformed(ActionEvent ae) {
        Object source = ae.getSource();

        if (source == startButton) {
            try {
                boolean success = startExport();

                if (success) {
                    closeDialog(null);
                } else {
                    // do nothing
                }
            } catch (Exception ee) {
                JOptionPane.showMessageDialog(this,
                    ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                    "(" + ee.getMessage() + ")",
                    ElanLocale.getString("Message.Error"),
                    JOptionPane.ERROR_MESSAGE);
                ee.printStackTrace();
            }
        } else if (source == closeButton) {
            closeDialog(null);
        }
    }

    /**
     * Starts the actual export after performing some checks.
     *
     * @return true if export succeeded, false oherwise
     *
     * @throws IOException DOCUMENT ME!
     */
    protected abstract boolean startExport() throws IOException;

    /**
     * Closes the dialog
     *
     * @param evt the window closing event
     */
    protected void closeDialog(WindowEvent evt) {
        setVisible(false);
        dispose();
    }

    /**
     * DOCUMENT ME!
     */
    protected void makeLayout() {
        setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);

        addWindowListener(new WindowAdapter() {
                public void windowClosing(WindowEvent evt) {
                    closeDialog(evt);
                }
            });

        titleLabel.setFont(titleLabel.getFont().deriveFont((float) 16));
        titleLabel.setHorizontalAlignment(SwingConstants.CENTER);

        buttonPanel.setLayout(new GridLayout(1, 2, 6, 0));

        startButton.addActionListener(this);
        buttonPanel.add(startButton);

        closeButton.addActionListener(this);
        buttonPanel.add(closeButton);
    }

    /**
     * Pack, size and set location.
     */
    protected void postInit() {
        pack();
        setSize((getSize().width < minimalWidth) ? minimalWidth : getSize().width,
            (getSize().height < minimalHeight) ? minimalHeight : getSize().height);
        setLocationRelativeTo(getParent());
        setResizable(false);
    }

    /**
     * Prompts the user for a file name and location.
     *
     * @param chooserTitle the title for the save dialog
     * @param extensions the file extensions (one of the constants of FileExtension)
     * @param filter the FileFilter(s) for the dialog
     * @param showEncodingBox if true, a combobox for selecting the encoding for the output file
     * @param encodings the list of encodings the user can choose from
     *
     * @return a file (unique) path
     */
    protected File promptForFile(String chooserTitle, String[] extensions,
        FileFilter filter, boolean showEncodingBox, String[] encodings) {
        JFileChooser chooser = null;

        if (showEncodingBox) {
            chooser = new FileAndEncodingChooser();

            if (encodings != null) {
                ((FileAndEncodingChooser) chooser).setEncodings(encodings);

                for (int i = encodings.length - 1; i >= 0; i--) {
                    if (FileAndEncodingChooser.UTF_8.equals(encodings[i])) {
                        ((FileAndEncodingChooser) chooser).setSelectedEncoding(encodings[i]);

                        break;
                    }

                    if (i == 0) {
                        ((FileAndEncodingChooser) chooser).setSelectedEncoding(encodings[0]);
                    }
                }
            } else {
                ((FileAndEncodingChooser) chooser).setSelectedEncoding(FileAndEncodingChooser.UTF_8);
            }
        } else {
            chooser = new JFileChooser();
        }

        if (transcription != null) {
            chooser.setSelectedFile(getDefaultExportFile(
                    transcription.getFullPath(), extensions[0]));
        } else {
            chooser.setSelectedFile(getDefaultExportFile(null, extensions[0]));
        }

        chooser.setDialogTitle(chooserTitle);

        chooser.setFileFilter(filter);

        if (chooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION) {
            setDefaultExportDir(chooser.getCurrentDirectory());

            File exportFile = chooser.getSelectedFile();

            if (showEncodingBox) {
                encoding = ((FileAndEncodingChooser) chooser).getSelectedEncoding();
            }

            String name = exportFile.getAbsolutePath();
            String lowerPathName = name.toLowerCase();

            boolean validExt = false;

            for (int i = 0; i < extensions.length; i++) {
                if (lowerPathName.endsWith("." + extensions[i].toLowerCase())) {
                    validExt = true;

                    break;
                }
            }

            if (!validExt) {
                name += ("." + extensions[0]);
                exportFile = new File(name);
            }

            if (exportFile.exists()) {
                int answer = JOptionPane.showConfirmDialog(null,
                        ElanLocale.getString("Message.Overwrite"),
                        ElanLocale.getString("SaveDialog.Message.Title"),
                        JOptionPane.YES_NO_OPTION);

                if (answer == JOptionPane.NO_OPTION) {
                    return promptForFile(chooserTitle, extensions, filter,
                        showEncodingBox);
                }
            }

            return exportFile;
        }

        return null;
    }

    /**
     * Prompts the user for a file name and location.
     *
     * @param chooserTitle the title for the save dialog
     * @param extensions the file extensions (one of the constants of FileExtension)
     * @param filter the FileFilter(s) for the dialog
     * @param showEncodingBox if true, a combobox for selecting the encoding for the output file
     *
     * @return a file (unique) path
     */
    protected File promptForFile(String chooserTitle, String[] extensions,
        FileFilter filter, boolean showEncodingBox) {
        return promptForFile(chooserTitle, extensions, filter, showEncodingBox,
            null);
    }

    /**
     * DOCUMENT ME!
     */
    protected void updateLocale() {
        startButton.setText(ElanLocale.getString("Button.OK"));
        closeButton.setText(ElanLocale.getString("Button.Close"));
    }

    /**
     * store last used export directory in properties
     *
     * @param directory
     */
    private static void setDefaultExportDir(File directory) {
        if (directory != null) {
            Preferences.set(LAST_USED_EXPORT_DIR, directory.getAbsolutePath(),
                null);
        }
    }

    /**
     * tries to find most appropiate default file
     *
     * @param transcriptionPath
     * @param extension
     *
     * @return export file
     */
    private static File getDefaultExportFile(String transcriptionPath,
        String extension) {
        String exportDir = (String) Preferences.get(LAST_USED_EXPORT_DIR, null);
        File file = null;

        if (transcriptionPath != null) {
            file = new File(transcriptionPath);
        }

        if ((exportDir == null) && (file != null)) {
            exportDir = file.getParent();
        }

        if (exportDir == null) {
            exportDir = System.getProperty("user.dir");
        }

        int index = -1;

        if (file != null) {
            index = file.getName().lastIndexOf('.');
        }

        String exportFileName = (index > -1)
            ? file.getName().substring(0, index)
            : ElanLocale.getString("Frame.ElanFrame.Untitled");

        return new File(exportDir, exportFileName + "." + extension);
    }
}
