/*
 * File:     ExportQtSubtitleDialog.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Selection;

import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.client.util.CheckBoxTableCellRenderer;
import mpi.eudico.client.util.Transcription2QtSubtitle;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;

import java.awt.Frame;
import java.awt.GridBagConstraints;

import java.io.File;
import java.io.IOException;

import java.util.List;
import java.util.Vector;

import javax.swing.DefaultCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JOptionPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;


/**
 * DOCUMENT ME! $Id: ExportQtSubtitleDialog.java 11133 2007-12-04 16:03:20Z hasloe $
 *
 * @author $Author: ericauer $
 * @version $Revision: 1.3 $
 */
public class ExportQtSubtitleDialog extends AbstractTierExportDialog
    implements ListSelectionListener, ChangeListener {
    private JCheckBox minimalDurCB;
    private JTextField minimalDurTF;
    private JCheckBox correctTimesCB;
    private JCheckBox mergeTiersCB;

    /**
     * DOCUMENT ME!
     *
     * @param parent
     * @param modal
     * @param transcription
     * @param selection
     */
    public ExportQtSubtitleDialog(Frame parent, boolean modal,
        Transcription transcription, Selection selection) {
        super(parent, modal, transcription, selection);
        makeLayout();
        extractTiers();
        postInit();
        restrictCheckBox.requestFocus();
    }

    /**
     * Extract candidate tiers for export.
     */
    protected void extractTiers() {
        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            if (transcription != null) {
                Vector v = transcription.getTiers();
                TierImpl t;

                for (int i = 0; i < v.size(); i++) {
                    t = (TierImpl) v.get(i);

                    // add all
                    if (i == 0) {
                        model.addRow(new Object[] { Boolean.TRUE, t.getName() });
                    } else {
                        model.addRow(new Object[] { Boolean.FALSE, t.getName() });
                    }
                }

                if (model.getRowCount() > 0) {
                    tierTable.setRowSelectionInterval(0, 0);
                }
            }
        }
    }

    /**
     * Initializes UI elements. Note: (for the time being) the checkbox column
     * indicating the selected state has been removed. It isn't that useful in
     * a single selection mode. The table could be replaced by a JList. When
     * the checkbox column would  be added again one of two things should
     * happen: either a custom tablecelleditor should be written for the
     * checkbox column or the valueChanged method should iterate over all rows
     * and uncheck all boxes, except the one for the selected roe.
     */
    protected void makeLayout() {
        super.makeLayout();
        model.setColumnIdentifiers(new String[] { EXPORT_COLUMN, TIER_NAME_COLUMN });

        tierTable.getColumn(EXPORT_COLUMN).setCellEditor(new DefaultCellEditor(
                new JCheckBox()));
        tierTable.getColumn(EXPORT_COLUMN).setCellRenderer(new CheckBoxTableCellRenderer());
        tierTable.getColumn(EXPORT_COLUMN).setMaxWidth(30);
        tierTable.setShowVerticalLines(false);
        tierTable.setTableHeader(null);

        //tierTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        tierTable.getSelectionModel().addListSelectionListener(this);

        // options
        GridBagConstraints gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;

        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        optionsPanel.add(restrictCheckBox, gridBagConstraints);

        correctTimesCB = new JCheckBox();
        correctTimesCB.setSelected(true);
        gridBagConstraints.gridy = 1;
        optionsPanel.add(correctTimesCB, gridBagConstraints);

        minimalDurCB = new JCheckBox();
        minimalDurCB.setSelected(false);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        optionsPanel.add(minimalDurCB, gridBagConstraints);

        minimalDurTF = new JTextField(6);
        minimalDurTF.setEnabled(false);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        optionsPanel.add(minimalDurTF, gridBagConstraints);

        mergeTiersCB = new JCheckBox();
        mergeTiersCB.setSelected(true);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        optionsPanel.add(mergeTiersCB, gridBagConstraints);

        minimalDurCB.addChangeListener(this);

        updateLocale();
    }

    /**
     * @see mpi.eudico.client.annotator.export.AbstractTierExportDialog#startExport()
     */
    protected boolean startExport() throws IOException {
        List selectedTiers = getSelectedTiers();

        if (selectedTiers.size() == 0) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportTradTranscript.Message.NoTiers"),
                ElanLocale.getString("Message.Warning"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        int minimalDur = 0;

        if (minimalDurCB.isSelected()) {
            String dur = minimalDurTF.getText();

            if ((dur == null) || (dur.length() == 0)) {
                JOptionPane.showMessageDialog(this,
                    ElanLocale.getString("ExportDialog.Message.InvalidNumber"),
                    ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);

                minimalDurTF.requestFocus();

                return false;
            }

            try {
                minimalDur = Integer.parseInt(dur);
            } catch (NumberFormatException nfe) {
                JOptionPane.showMessageDialog(this,
                    ElanLocale.getString("ExportDialog.Message.InvalidNumber"),
                    ElanLocale.getString("Message.Warning"),
                    JOptionPane.WARNING_MESSAGE);

                minimalDurTF.requestFocus();

                return false;
            }
        }

        // prompt for file name and location
        File exportFile = promptForFile(ElanLocale.getString(
                    "ExportQtSubtitleDialog.Title"), FileExtension.TEXT_EXT,
                ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE), false);

        if (exportFile == null) {
            return false;
        }

        long offset = 0L;

        if (correctTimesCB.isSelected()) {
            Vector mediaDescriptors = transcription.getMediaDescriptors();

            if (mediaDescriptors.size() > 0) {
                offset = ((MediaDescriptor) mediaDescriptors.get(0)).timeOrigin;
            }
        }

        boolean merge = mergeTiersCB.isSelected();
        long mediaDur = ELANCommandFactory.getViewerManager(transcription)
                                          .getMasterMediaPlayer()
                                          .getMediaDuration();

        String[] tierNames = (String[]) selectedTiers.toArray(new String[0]);
        long b = 0L;
        long e = Long.MAX_VALUE;

        if (restrictCheckBox.isSelected()) {
            b = selection.getBeginTime();
            e = selection.getEndTime();
        }

        if (!merge) {
            Transcription2QtSubtitle.exportTiers(transcription, tierNames,
                exportFile, b, e, offset, minimalDur, mediaDur);
        } else {
            Transcription2QtSubtitle.exportTiersMerged(transcription,
                tierNames, exportFile, b, e, offset, minimalDur, mediaDur);
        }

        return true;
    }

    /**
     * @see mpi.eudico.client.annotator.export.AbstractTierExportDialog#updateLocale()
     */
    protected void updateLocale() {
        super.updateLocale();
        setTitle(ElanLocale.getString("ExportQtSubtitleDialog.Title"));
        titleLabel.setText(ElanLocale.getString(
                "ExportQtSubtitleDialog.TitleLabel"));
        correctTimesCB.setText(ElanLocale.getString("ExportDialog.CorrectTimes"));
        minimalDurCB.setText(ElanLocale.getString(
                "ExportDialog.Label.MinimalDur"));
        mergeTiersCB.setText(ElanLocale.getString(
                "ExportQtSubtitleDialog.Label.Merge"));
    }

    /**
     * Updates the checked state of the export checkboxes.
     *
     * @param lse the list selection event
     */
    public void valueChanged(ListSelectionEvent lse) {
        if ((model != null) && lse.getValueIsAdjusting()) {
            int b = lse.getFirstIndex();
            int e = lse.getLastIndex();
            int col = model.findColumn(EXPORT_COLUMN);

            for (int i = b; i <= e; i++) {
                if (tierTable.isRowSelected(i)) {
                    model.setValueAt(Boolean.TRUE, i, col);
                }
            }
        }
    }

    /**
     * Enables/disables the minimal duration textfield.
     *
     * @param e the change event
     */
    public void stateChanged(ChangeEvent e) {
        if (e.getSource() == minimalDurCB) {
            minimalDurTF.setEnabled(minimalDurCB.isSelected());
        }
    }
}
