/*
 * File:     ExportToolboxDialog.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.gui.ShoeboxMarkerDialog;

import mpi.eudico.client.annotator.interlinear.Interlinear;
import mpi.eudico.client.annotator.interlinear.ToolboxEncoder;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.client.util.CheckBoxTableCellRenderer;
import mpi.eudico.client.util.SelectableObject;

import mpi.eudico.server.corpora.clomimpl.abstr.MediaDescriptor;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.shoebox.MarkerRecord;
import mpi.eudico.server.corpora.clomimpl.shoebox.ShoeboxTypFile;
import mpi.eudico.server.corpora.clomimpl.shoebox.ToolboxEncoderInfo;
import mpi.eudico.server.corpora.clomimpl.type.Constraint;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.event.WindowEvent;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.DefaultCellEditor;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.DefaultTableCellRenderer;


/**
 * An export dialog for exporting tiers to a Shoebox/Toolbox file.
 *
 * @author Han Sloetjes
 */
public class ExportToolboxDialog extends AbstractTierExportDialog
    implements ActionListener, ItemListener, ListSelectionListener {
    private JButton downButton;
    private JButton typButton;
    private JButton upButton;
    private JCheckBox blankLineCB;

    //private JCheckBox allUnicodeCB;
    private JCheckBox correctTimesCB;
    private JCheckBox wrapBlocksCB;
    private JCheckBox wrapLinesCB;
    private JCheckBox includeEmptyLinesCB;

    /** ui elements */
    private JLabel charPerLineLabel;
    private JLabel toolboxDBTypeLabel;
    private JLabel recordMarkerLabel;
    private JLabel timeFormatLabel;
    private JPanel markerPanel;
    private JRadioButton hhMMSSMSFormatRB;
    private JRadioButton specRB;
    private JRadioButton ssMSFormatRB;
    private JRadioButton typeRB;
    private JRadioButton wrapNextLineRB;
    private JRadioButton wrapAfterBlockRB;
    private JRadioButton detectedRMRB;
    private JRadioButton defaultRMRB;
    private JRadioButton customRMRB;
    private JTextField dbTypField;
    private JTextField numCharTF;
    private JTextField typField;
    private JTextField markerTF;

    // some strings
    // not visible in the table header

    /** default line width */
    private final int NUM_CHARS = 80;
    private List markers;

    // fields for the encoder
    private String databaseType;
    private String exportFileName;

    // count the number of root tiers after 'collapsing' or 'merging' all 
    // 'marker@part' tiers to 'marker'
    private int numRootTiers = 1;
    private String recordMarker = "";
    private ArrayList mergedTiers;
    private ArrayList markersWithBlankLine;

    /** Holds value of property DOCUMENT ME! */
    private final String elanBeginLabel = "ELANBegin";

    /** Holds value of property DOCUMENT ME! */
    private final String elanEndLabel = "ELANEnd";

    /** Holds value of property DOCUMENT ME! */
    private final String elanParticipantLabel = "ELANParticipant";

    /**
     * Constructor.
     *
     * @param parent parent frame
     * @param modal the modal/blocking attribute
     * @param transcription the transcription to export from
     */
    public ExportToolboxDialog(Frame parent, boolean modal,
        TranscriptionImpl transcription) {
        super(parent, modal, transcription, null);
        mergedTiers = new ArrayList();
        markersWithBlankLine = new ArrayList(5);
        makeLayout();
        extractTiers();
        postInit();
        typField.requestFocus();
    }

    /**
     *
     *
     * @param tier DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    public Vector getTierTree(TierImpl tier) {
        Vector tierTree = new Vector();
        Vector tierTrees = new Vector();

        Vector children = tier.getChildTiers();

        tierTree.add(tier);

        for (int j = 0; j < children.size(); j++) {
            TierImpl child = (TierImpl) children.elementAt(j);
            tierTrees.add(getTierTree(child));
        }

        Collections.sort(tierTrees, new VectorComparator());

        for (int j = 0; j < tierTrees.size(); j++) {
            tierTree.addAll((Vector) tierTrees.elementAt(j));
        }

        return tierTree;
    }

    /**
     * The action performed event handling.
     *
     * @param ae the action event
     */
    public void actionPerformed(ActionEvent ae) {
        Object source = ae.getSource();

        if (source == upButton) {
            moveUp();
        } else if (source == downButton) {
            moveDown();
        } else if (source == typButton) {
            chooseTyp();
            typeRB.setSelected(true);
        } else {
            super.actionPerformed(ae);
        }
    }

    /**
     * The item state changed handling.
     *
     * @param ie the ItemEvent
     */
    public void itemStateChanged(ItemEvent ie) {
        if (ie.getSource() == wrapBlocksCB) {
            if (wrapBlocksCB.isSelected()) {
                setDefaultNumOfChars();
                numCharTF.requestFocus();
                wrapLinesCB.setEnabled(true);
                wrapNextLineRB.setEnabled(wrapLinesCB.isSelected());
                wrapAfterBlockRB.setEnabled(wrapLinesCB.isSelected());
            } else {
                numCharTF.setEnabled(false);
                numCharTF.setBackground(Constants.DEFAULTBACKGROUNDCOLOR);
                wrapLinesCB.setEnabled(false);
                wrapNextLineRB.setEnabled(false);
                wrapAfterBlockRB.setEnabled(false);
            }
        } else if (ie.getSource() == wrapLinesCB) {
            wrapNextLineRB.setEnabled(wrapLinesCB.isSelected());
            wrapAfterBlockRB.setEnabled(wrapLinesCB.isSelected());
        } else if (ie.getSource() == typeRB) {
            enableTypComponents(true);
        } else if (ie.getSource() == specRB) {
            enableTypComponents(false);
        } else if (ie.getSource() == blankLineCB) {
            int row = tierTable.getSelectedRow();

            if (row > -1) {
                Object val = model.getValueAt(row,
                        model.findColumn(TIER_NAME_COLUMN));

                if (val instanceof SelectableObject) {
                    ((SelectableObject) val).setSelected(blankLineCB.isSelected());
                    tierTable.repaint();
                }
            }
        } else if ((ie.getSource() == detectedRMRB) ||
                (ie.getSource() == defaultRMRB)) {
            markerTF.setEnabled(false);
        } else if (ie.getSource() == customRMRB) {
            markerTF.setEnabled(true);
        }
    }

    /**
     * Updates the checked state of the export checkboxes.
     *
     * @param lse the list selection event
     */
    public void valueChanged(ListSelectionEvent lse) {
        if ((model != null) && lse.getValueIsAdjusting()) {
            int col = model.findColumn(EXPORT_COLUMN);
            int row = tierTable.getSelectedRow();

            if (row > -1) {
                if (tierTable.isRowSelected(row)) {
                    model.setValueAt(Boolean.TRUE, row, col);
                }

                Object val = model.getValueAt(row,
                        model.findColumn(TIER_NAME_COLUMN));

                if (val instanceof SelectableObject) {
                    blankLineCB.setSelected(((SelectableObject) val).isSelected());
                }
            }
        }
    }

    /**
     * Extract candidate tiers for export.
     */
    protected void extractTiers() {
        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            if (transcription != null) {
                Vector v = transcription.getTiers();
                TierImpl t;

                ArrayList rootTiers = new ArrayList(5);
                String tName;
                String markName;

                for (int i = 0; i < v.size(); i++) {
                    t = (TierImpl) v.get(i);
                    tName = t.getName();

                    int atIndex = tName.indexOf('@');

                    if (atIndex > -1) {
                        markName = tName.substring(0, atIndex);

                        if (!mergedTiers.contains(markName)) {
                            mergedTiers.add(markName);
                        }

                        if (!t.hasParentTier()) {
                            if (!rootTiers.contains(markName)) {
                                rootTiers.add(markName);
                            }
                        }
                    } else {
                        mergedTiers.add(tName);

                        if (!t.hasParentTier()) {
                            rootTiers.add(tName);
                        }
                    }
                }

                numRootTiers = rootTiers.size();

                if (numRootTiers == 1) {
                    recordMarker = (String) rootTiers.get(0);

                    int index = mergedTiers.indexOf(recordMarker);

                    if (index != 0) {
                        mergedTiers.remove(index);
                        mergedTiers.add(0, recordMarker);
                    }

                    if (detectedRMRB != null) {
                        detectedRMRB.setEnabled(true);
                        detectedRMRB.setText(detectedRMRB.getText() + " (\\" +
                            recordMarker + ")");
                    }
                } else {
                    detectedRMRB.setEnabled(false);
                    defaultRMRB.setSelected(true);
                }

                if (mergedTiers.size() > 1) {
                    mergedTiers.add(1, elanParticipantLabel);
                    mergedTiers.add(1, elanEndLabel);
                    mergedTiers.add(1, elanBeginLabel);
                } else {
                    mergedTiers.add(elanBeginLabel);
                    mergedTiers.add(elanEndLabel);
                    mergedTiers.add(elanParticipantLabel);
                }

                for (int i = 0; i < mergedTiers.size(); i++) {
                    model.addRow(new Object[] {
                            Boolean.TRUE,
                            new SelectableObject(mergedTiers.get(i), false)
                        });
                }
            }

            if (model.getRowCount() > 1) {
                upButton.setEnabled(true);
                downButton.setEnabled(true);
            } else {
                upButton.setEnabled(false);
                downButton.setEnabled(false);
            }
        } else {
            upButton.setEnabled(false);
            downButton.setEnabled(false);
        }
    }

    /**
     * Initializes UI elements.
     */
    protected void makeLayout() {
        super.makeLayout();
        charPerLineLabel = new JLabel();
        wrapBlocksCB = new JCheckBox();
        wrapBlocksCB.setSelected(true);
        numCharTF = new JTextField(4);
        timeFormatLabel = new JLabel();
        ssMSFormatRB = new JRadioButton();
        hhMMSSMSFormatRB = new JRadioButton();
        correctTimesCB = new JCheckBox();
        upButton = new JButton();
        downButton = new JButton();
        blankLineCB = new JCheckBox();
        blankLineCB.addItemListener(this);
        wrapLinesCB = new JCheckBox();
        wrapLinesCB.setSelected(true);
        wrapNextLineRB = new JRadioButton();
        wrapAfterBlockRB = new JRadioButton();
        wrapNextLineRB.setSelected(true);

        ButtonGroup wrapGroup = new ButtonGroup();
        wrapGroup.add(wrapNextLineRB);
        wrapGroup.add(wrapAfterBlockRB);
        includeEmptyLinesCB = new JCheckBox();
        includeEmptyLinesCB.setSelected(true);

        toolboxDBTypeLabel = new JLabel();
        typField = new JTextField("", 23);
        typButton = new JButton("...");

        //allUnicodeCB = new JCheckBox();
        dbTypField = new JTextField("", 14);

        ButtonGroup buttonGroup = new ButtonGroup();
        typeRB = new JRadioButton();
        typeRB.setSelected(true);
        typeRB.addItemListener(this);
        specRB = new JRadioButton();
        specRB.addItemListener(this);
        buttonGroup.add(typeRB);
        buttonGroup.add(specRB);
        recordMarkerLabel = new JLabel();
        detectedRMRB = new JRadioButton();
        detectedRMRB.setSelected(true);
        defaultRMRB = new JRadioButton();
        customRMRB = new JRadioButton();

        ButtonGroup rmGroup = new ButtonGroup();
        rmGroup.add(detectedRMRB);
        rmGroup.add(defaultRMRB);
        rmGroup.add(customRMRB);
        markerTF = new JTextField("", 6);
        markerTF.setEnabled(false);

        try {
            ImageIcon upIcon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/navigation/Up16.gif"));
            ImageIcon downIcon = new ImageIcon(this.getClass().getResource("/toolbarButtonGraphics/navigation/Down16.gif"));
            upButton.setIcon(upIcon);
            downButton.setIcon(downIcon);
        } catch (Exception ex) {
            upButton.setText("Up");
            downButton.setText("Down");
        }

        model.setColumnIdentifiers(new String[] { EXPORT_COLUMN, TIER_NAME_COLUMN });
        tierTable.getColumn(EXPORT_COLUMN).setCellEditor(new DefaultCellEditor(
                new JCheckBox()));
        tierTable.getColumn(EXPORT_COLUMN).setCellRenderer(new CheckBoxTableCellRenderer());
        tierTable.getColumn(TIER_NAME_COLUMN).setCellRenderer(new MarkerCellRenderer());
        tierTable.getColumn(EXPORT_COLUMN).setMaxWidth(30);
        tierTable.setShowVerticalLines(false);
        tierTable.setTableHeader(null);
        tierTable.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        tierTable.getSelectionModel().addListSelectionListener(this);

        GridBagConstraints gridBagConstraints;
        Insets vertInsets = new Insets(0, 6, 2, 6);

        upButton.addActionListener(this);
        downButton.addActionListener(this);

        JPanel updownPanel = new JPanel(new GridBagLayout());

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        updownPanel.add(upButton, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        updownPanel.add(downButton, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        updownPanel.add(new JPanel(), gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.EAST;
        gridBagConstraints.insets = insets;
        updownPanel.add(blankLineCB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        tierSelectionPanel.add(updownPanel, gridBagConstraints);

        optionsPanel.setLayout(new GridBagLayout());

        JPanel wrapPanel = new JPanel(new GridBagLayout());
        JPanel timePanel = new JPanel(new GridBagLayout());

        wrapBlocksCB.addItemListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        wrapPanel.add(wrapBlocksCB, gridBagConstraints);

        JPanel fill = new JPanel();
        Dimension fillDim = new Dimension(30, 10);
        fill.setPreferredSize(fillDim);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        wrapPanel.add(fill, gridBagConstraints);

        numCharTF.setEnabled(false);
        numCharTF.setBackground(Constants.DEFAULTBACKGROUNDCOLOR);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        wrapPanel.add(numCharTF, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        wrapPanel.add(charPerLineLabel, gridBagConstraints);

        wrapLinesCB.addItemListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        wrapPanel.add(wrapLinesCB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        wrapPanel.add(wrapNextLineRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        wrapPanel.add(wrapAfterBlockRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        wrapPanel.add(includeEmptyLinesCB, gridBagConstraints);

        // time
        ButtonGroup group = new ButtonGroup();
        group.add(ssMSFormatRB);
        ssMSFormatRB.setSelected(true);
        group.add(hhMMSSMSFormatRB);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        timePanel.add(timeFormatLabel, gridBagConstraints);

        fill = new JPanel();
        fillDim = new Dimension(30, 10);
        fill.setPreferredSize(fillDim);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        timePanel.add(fill, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        timePanel.add(hhMMSSMSFormatRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        timePanel.add(ssMSFormatRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = insets;
        timePanel.add(correctTimesCB, gridBagConstraints);

        // add to options panel
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.NORTHWEST;
        optionsPanel.add(wrapPanel, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        optionsPanel.add(new JPanel(), gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = GridBagConstraints.NORTHEAST;
        optionsPanel.add(timePanel, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = insets;
        getContentPane().add(optionsPanel, gridBagConstraints);

        markerPanel = new JPanel();
        markerPanel.setLayout(new GridBagLayout());

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = insets;
        markerPanel.add(toolboxDBTypeLabel, gridBagConstraints);

        JPanel fill2 = new JPanel();
        fill2.setPreferredSize(fillDim);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(fill2, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(typeRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(typField, gridBagConstraints);

        typButton.addActionListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 3;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(typButton, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(specRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(dbTypField, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.NORTHWEST;
        gridBagConstraints.insets = insets;
        markerPanel.add(recordMarkerLabel, gridBagConstraints);

        JPanel fill3 = new JPanel();
        fill3.setPreferredSize(fillDim);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.gridheight = 3;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(fill3, gridBagConstraints);

        detectedRMRB.addItemListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(detectedRMRB, gridBagConstraints);

        defaultRMRB.addItemListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(defaultRMRB, gridBagConstraints);

        customRMRB.addItemListener(this);
        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(customRMRB, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 6;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.insets = vertInsets;
        markerPanel.add(markerTF, gridBagConstraints);

        gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = GridBagConstraints.WEST;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = insets;
        getContentPane().add(markerPanel, gridBagConstraints);

        //move buttonPanel from 3rd to 4th component
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = GridBagConstraints.NONE;
        gridBagConstraints.anchor = GridBagConstraints.CENTER;
        ((GridBagLayout) getContentPane().getLayout()).setConstraints(buttonPanel,
            gridBagConstraints);

        setDefaultNumOfChars();

        setShoeboxMarkerRB();

        updateLocale();
    }

    /**
     * Shows a warning/error dialog with the specified message string.
     *
     * @param message the message to display
     */
    protected void showWarningDialog(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Starts the actual export after performing some checks.
     *
     * @return true if export succeeded, false oherwise
     */
    protected boolean startExport() {
        if (!checkFields()) {
            return false;
        }

        List selectedTiers = getSelectedTiers();

        if (selectedTiers.size() == 0) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportTradTranscript.Message.NoTiers"),
                ElanLocale.getString("Message.Warning"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        // check the chars per line value
        int charsPerLine = Integer.MAX_VALUE;

        if (wrapBlocksCB.isSelected()) {
            String textValue = numCharTF.getText().trim();

            try {
                charsPerLine = Integer.parseInt(textValue);
            } catch (NumberFormatException nfe) {
                showWarningDialog(ElanLocale.getString(
                        "ExportShoebox.Message.InvalidNumber"));
                numCharTF.selectAll();
                numCharTF.requestFocus();

                return false;
            }
        }

        int timeFormat = Interlinear.SSMS;

        if (hhMMSSMSFormatRB.isSelected()) {
            timeFormat = Interlinear.HHMMSSMS;
        }

        // prompt for file name and location
        File exportFile = promptForFile(ElanLocale.getString(
                    "ExportShoebox.Title.Toolbox"), FileExtension.TEXT_EXT,
                ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE), false);

        if (exportFile == null) {
            return false;
        }

        exportFileName = exportFile.getPath();

        // export....
        boolean success = doExport(exportFileName, selectedTiers, charsPerLine,
                timeFormat, correctTimesCB.isSelected());

        return success;
    }

    /**
     * Applies localized strings to the ui elements. For historic reasons the
     * string identifiers start with "TokenizeDialog"
     */
    protected void updateLocale() {
        super.updateLocale();
        setTitle(ElanLocale.getString("ExportShoebox.Title.Toolbox"));
        titleLabel.setText(ElanLocale.getString("ExportShoebox.Title.Toolbox"));
        blankLineCB.setText(ElanLocale.getString(
                "ExportShoebox.Button.BlankLineAfter"));
        markerPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "ExportShoebox.Label.ToolboxOptions")));
        wrapBlocksCB.setText(ElanLocale.getString(
                "ExportShoebox.Label.WrapBlocks"));
        charPerLineLabel.setText(ElanLocale.getString(
                "ExportShoebox.Label.NumberChars"));
        wrapLinesCB.setText(ElanLocale.getString(
                "ExportShoebox.Label.WrapLines"));
        wrapAfterBlockRB.setText(ElanLocale.getString(
                "ExportShoebox.Label.WrapEndOfBlock"));
        wrapNextLineRB.setText(ElanLocale.getString(
                "ExportShoebox.Label.WrapNextLine"));
        includeEmptyLinesCB.setText(ElanLocale.getString(
                "ExportShoebox.Label.IncludeEmpty"));
        timeFormatLabel.setText(ElanLocale.getString(
                "ExportShoebox.Label.Format"));
        hhMMSSMSFormatRB.setText(ElanLocale.getString(
                "InterlinearizerOptionsDlg.TimeCodeFormat.TimeCode"));
        ssMSFormatRB.setText(ElanLocale.getString(
                "InterlinearizerOptionsDlg.TimeCodeFormat.Seconds"));
        correctTimesCB.setText(ElanLocale.getString("ExportDialog.CorrectTimes"));
        toolboxDBTypeLabel.setText(ElanLocale.getString(
                "ExportShoebox.Label.ToolboxBDName"));
        toolboxDBTypeLabel.setToolTipText("e.g. \\_sh v3.0  400 Text");
        typeRB.setText(ElanLocale.getString("ExportShoebox.Label.Type"));

        //allUnicodeCB.setText(ElanLocale.getString(
        //        "ExportShoebox.CheckBox.AllUnicode"));
        //fieldSpecButton.setText(ElanLocale.getString(
        //        "ExportShoebox.Button.FieldSpec"));
        specRB.setText(ElanLocale.getString("ExportShoebox.Label.SpecifyType"));

        //tierNamesLabel.setText(ElanLocale.getString(
        //        "ExportShoebox.Label.UseTierNames"));
        //generateMarkersCB.setText(ElanLocale.getString(
        //        "ExportShoebox.CheckBox.AutoGenerateMarkers"));
        recordMarkerLabel.setText(ElanLocale.getString(
                "ExportShoebox.Label.RecordMarker"));
        detectedRMRB.setText(ElanLocale.getString(
                "ExportShoebox.Label.Detected"));
        defaultRMRB.setText(ElanLocale.getString(
                "ExportShoebox.Label.DefaultMarker") + " (\\block)");
        customRMRB.setText(ElanLocale.getString(
                "ExportShoebox.Label.CustomMarker"));
    }

    private void setDefaultNumOfChars() {
        numCharTF.setEnabled(true);
        numCharTF.setBackground(Constants.SHAREDCOLOR4);

        if ((numCharTF.getText() != null) ||
                (numCharTF.getText().length() == 0)) {
            numCharTF.setText("" + NUM_CHARS);
        }
    }

    private List getMarkersWithBlankLines() {
        List mbl = new ArrayList();
        int nameCol = model.findColumn(TIER_NAME_COLUMN);

        // add selected tiers in the right order
        for (int i = 0; i < model.getRowCount(); i++) {
            SelectableObject sob = (SelectableObject) model.getValueAt(i,
                    nameCol);

            if (sob.isSelected()) {
                mbl.add(sob.getValue());
            }
        }

        return mbl;
    }

    /*
        private void setEnabledAllUnicode(boolean enable) {
            allUnicodeCB.setSelected(false);
            allUnicodeCB.setEnabled(enable);
        }

        private void setEnabledAutoGenerate(boolean enable) {
            generateMarkersCB.setSelected(false);
            generateMarkersCB.setEnabled(enable);
        }
    */
    private Vector getHierarchicallySortedTiers(TranscriptionImpl transcription) {
        // for each root tier, find dependency tree.
        // store in a Vector with Vectors, one for each root.
        // take the largest tier tree first, this is likely to be the interlinear tree
        Vector tierTrees = new Vector();
        Vector sortedTiers = new Vector();

        Vector topTiers = transcription.getTopTiers();

        for (int i = 0; i < topTiers.size(); i++) {
            TierImpl topTier = (TierImpl) topTiers.elementAt(i);
            tierTrees.add(getTierTree(topTier));
        }

        Collections.sort(tierTrees, new VectorComparator());

        for (int j = 0; j < tierTrees.size(); j++) {
            sortedTiers.addAll((Vector) tierTrees.elementAt(j));
        }

        return sortedTiers;
    }

    private void setShoeboxMarkerRB() {
        Object useTyp = Preferences.get("LastUsedShoeboxExport", null);

        if ((useTyp == null) ||
                (useTyp instanceof String &&
                ((String) useTyp).equalsIgnoreCase("typ"))) {
            typeRB.setSelected(true);

            Object luTypFile = Preferences.get("LastUsedShoeboxTypFile", null);

            if (luTypFile instanceof String) {
                typField.setText((String) luTypFile);
            }

            enableTypComponents(true);
        } else {
            specRB.setSelected(true);
            enableTypComponents(false);
        }
    }

    private void enableTypComponents(boolean enable) {
        typField.setEnabled(enable);
        typButton.setEnabled(enable);
        dbTypField.setEnabled(!enable);
    }

    private void autoGenerateMarkerFile() {
        // generate marker records for each tier.
        // only marker, parent marker and stereotype have to be set, rest is default
        Vector markerRecords = new Vector();

        try {
            Vector tiers = transcription.getTiers();

            for (int i = 0; i < tiers.size(); i++) {
                TierImpl t = (TierImpl) tiers.elementAt(i);

                MarkerRecord mkrRecord = new MarkerRecord();
                mkrRecord.setMarker(t.getName());

                if (t.hasParentTier()) {
                    mkrRecord.setParentMarker(t.getParentTier().getName());

                    if (t.getLinguisticType() != null) {
                        int stereotype = t.getLinguisticType().getConstraints()
                                          .getStereoType();

                        if ((stereotype == Constraint.SYMBOLIC_SUBDIVISION) ||
                                (stereotype == Constraint.TIME_SUBDIVISION) ||
                                (stereotype == Constraint.INCLUDED_IN)) {
                            //mkrRecord.setStereoType(Constraint.publicStereoTypes[2]);
                            mkrRecord.setStereoType(Constraint.stereoTypes[Constraint.SYMBOLIC_SUBDIVISION]);
                        } else if (stereotype == Constraint.SYMBOLIC_ASSOCIATION) {
                            //mkrRecord.setStereoType(Constraint.publicStereoTypes[3]);
                            mkrRecord.setStereoType(Constraint.stereoTypes[Constraint.SYMBOLIC_ASSOCIATION]);
                        }
                    }
                }

                mkrRecord.setCharset(MarkerRecord.UNICODESTRING);
                mkrRecord.setParticipantMarker(false);
                mkrRecord.setExcluded(false);

                markerRecords.add(mkrRecord);
            }

            // store in mkr file with name of transcription, next to eaf
            // dec 2006 HS: by default the .mkr file will now be saved next to the export file
            String fileName = ((TranscriptionImpl) transcription).getPathName();

            if (exportFileName != null) {
                fileName = exportFileName.substring(0,
                        exportFileName.lastIndexOf("."));
            } else if (fileName.toLowerCase().endsWith(".eaf")) {
                fileName = fileName.substring(0, fileName.lastIndexOf("."));
            }

            fileName += ".mkr";

            final File newSaveFile = new File(fileName);

            if (newSaveFile != null) {
                if (newSaveFile.exists()) {
                    int answer = JOptionPane.showConfirmDialog(null,
                            ElanLocale.getString("Message.Overwrite") + "\n" +
                            fileName,
                            ElanLocale.getString("SaveDialog.Message.Title"),
                            JOptionPane.YES_NO_OPTION);

                    if (answer == JOptionPane.NO_OPTION) {
                        return;
                    }
                }

                FileOutputStream out = new FileOutputStream(newSaveFile);
                BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(
                            out, "UTF-8"));

                Iterator markerIter = markerRecords.iterator();

                while (markerIter.hasNext()) {
                    writer.write(((MarkerRecord) markerIter.next()).toString());
                }

                writer.close();
            }
        } catch (Exception ex) {
            ex.printStackTrace();
        }
    }

    /**
     * Checks the contents of marker input fields and next the existence of the
     * designated files.
     *
     * @return true if the files exist, false otherwise
     */
    private boolean checkFields() {
        // database type check
        if (typeRB.isSelected() &&
                ((typField.getText() == null) ||
                (typField.getText().length() == 0))) {
            showError(ElanLocale.getString("ImportDialog.Message.SpecifyType"));

            return false;
        }

        if (typeRB.isSelected()) {
            File tf = new File(typField.getText());

            if (!tf.exists()) {
                showError(ElanLocale.getString("ImportDialog.Message.NoType"));

                return false;
            } else {
                try {
                    ShoeboxTypFile typFile = new ShoeboxTypFile(tf);
                    databaseType = typFile.getDatabaseType();
                } catch (Exception e) {
                }
            }
        } else {
            databaseType = dbTypField.getText();

            if ((databaseType == null) || (databaseType.trim().length() == 0)) {
                showError(ElanLocale.getString("ExportShoebox.Message.NoType"));
                dbTypField.requestFocus();

                return false;
            }
        }

        // record marker test
        if (customRMRB.isSelected()) {
            String custRM = markerTF.getText();

            if ((custRM == null) || (custRM.trim().length() == 0)) {
                showError(ElanLocale.getString(
                        "ExportShoebox.Message.NoRecordMarker"));
                markerTF.requestFocus();

                return false;
            } else {
                recordMarker = custRM.trim();
            }
        } else if (defaultRMRB.isSelected()) {
            recordMarker = "block"; // should be a constant from elsewhere
        }
         // otherwise the record marker has been detected from the transcription

        return true;
    }

    private void chooseTyp() {
        String lastUsedDir = (String) Preferences.get("LastUsedShoeboxTypDir",
                null);

        if (lastUsedDir == null) {
            lastUsedDir = System.getProperty("user.dir");
        }

        JFileChooser chooser = new JFileChooser(lastUsedDir);
        chooser.setDialogTitle(ElanLocale.getString("ImportDialog.Title.Select"));
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.SHOEBOX_TYP_TYPE));
        chooser.setApproveButtonText(ElanLocale.getString(
                "ImportDialog.Approve"));
        chooser.setDialogType(JFileChooser.OPEN_DIALOG);

        int option = chooser.showDialog(this, null);

        if (option == JFileChooser.APPROVE_OPTION) {
            File curDir = chooser.getCurrentDirectory();

            if (curDir != null) {
                Preferences.set("LastUsedShoeboxTypDir",
                    curDir.getAbsolutePath(), null);
            }

            File f = chooser.getSelectedFile();

            if (f != null) {
                typField.setText(f.getAbsolutePath());
            }
        }
    }

    //******************************
    // actual export methods from here, for the time being
    //******************************

    /**
     * The actual writing.
     *
     * @param fileName path to the file, not null
     * @param orderedTiers tier names, ordered by the user, min size 1
     * @param charsPerLine num of chars per line if linewrap is selected
     * @param timeFormat the time format, a constant from Interlinear
     * @param correctTimes if true the master media time offset will be
     * added to all time values
     *
     * @return true if all went well, false otherwise
     */
    private boolean doExport(final String fileName, final List orderedTiers,
        final int charsPerLine, final int timeFormat, final boolean correctTimes) {
        int markerSource = ToolboxEncoderInfo.TIERNAMES; // default

        if (typeRB.isSelected()) {
            markerSource = ToolboxEncoderInfo.TYPFILE;
            Preferences.set("LastUsedShoeboxExport", "typ", null);
            Preferences.set("LastUsedShoeboxTypFile", typField.getText(), null);
        } else {
            Preferences.set("LastUsedShoeboxExport", "", null);
        }

        ToolboxEncoderInfo tbEncoderInfo = new ToolboxEncoderInfo(charsPerLine,
                markerSource, timeFormat);
        tbEncoderInfo.setCorrectAnnotationTimes(correctTimes);

        if (databaseType != null) {
            tbEncoderInfo.setDatabaseType(databaseType);
        }

        // the new options
        if (charsPerLine != Integer.MAX_VALUE) {
            tbEncoderInfo.setWrapLines(wrapLinesCB.isSelected());

            if (wrapLinesCB.isSelected()) {
                if (wrapNextLineRB.isSelected()) {
                    tbEncoderInfo.setLineWrapStyle(Interlinear.NEXT_LINE);
                } else {
                    tbEncoderInfo.setLineWrapStyle(Interlinear.END_OF_BLOCK);
                }
            } else {
                tbEncoderInfo.setLineWrapStyle(Interlinear.NO_WRAP);
            }
        } else {
            // no block and no line wrapping
            tbEncoderInfo.setWrapLines(false);
            tbEncoderInfo.setLineWrapStyle(Interlinear.NO_WRAP);
        }

        if (correctTimesCB.isSelected()) {
            List mds = transcription.getMediaDescriptors();

            if ((mds != null) && (mds.size() > 0)) {
                long mediaOffset = ((MediaDescriptor) mds.get(0)).timeOrigin;
                tbEncoderInfo.setTimeOffset(mediaOffset);
            }
        }

        tbEncoderInfo.setIncludeEmptyMarkers(includeEmptyLinesCB.isSelected());
        tbEncoderInfo.setRecordMarker(recordMarker);
        tbEncoderInfo.setOrderedVisibleTiers(orderedTiers);
        tbEncoderInfo.setMarkersWithBlankLines(getMarkersWithBlankLines());

        if (fileName != null) {
            try {
                ToolboxEncoder encoder = new ToolboxEncoder();
                encoder.encodeAndSave(transcription, tbEncoderInfo,
                    orderedTiers, fileName);
            } catch (IOException ioe) {
                JOptionPane.showMessageDialog(this,
                    ElanLocale.getString("ExportDialog.Message.Error") + "\n" +
                    "(" + ioe.getMessage() + ")",
                    ElanLocale.getString("Message.Error"),
                    JOptionPane.ERROR_MESSAGE);
            }
        }

        return true;
    }

    /**
     * Moves selected tiers up in the list of tiers.
     */
    private void moveDown() {
        if ((tierTable == null) || (model == null) ||
                (model.getRowCount() < 2)) {
            return;
        }

        int row = tierTable.getSelectedRow();

        if (row > -1) {
            if ((row == 0) && (numRootTiers == 1)) {
                return;
            }

            if ((row < (model.getRowCount() - 1)) &&
                    !tierTable.isRowSelected(row + 1)) {
                model.moveRow(row, row, row + 1);
                tierTable.changeSelection(row, 0, true, false);
                tierTable.changeSelection(row + 1, 0, true, false);
            }
        }
    }

    /**
     * Moves selected tiers up in the list of tiers.
     */
    private void moveUp() {
        if ((tierTable == null) || (model == null) ||
                (model.getRowCount() < 2)) {
            return;
        }

        int row = tierTable.getSelectedRow();

        if (row > -1) {
            if ((row == 1) && (numRootTiers == 1)) {
                return;
            }

            if ((row > 0) && !tierTable.isRowSelected(row - 1)) {
                model.moveRow(row, row, row - 1);
                tierTable.changeSelection(row, 0, true, false);
                tierTable.changeSelection(row - 1, 0, true, false);
            }
        }
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected List getSelectedTiers() {
        int includeCol = model.findColumn(EXPORT_COLUMN);
        int nameCol = model.findColumn(TIER_NAME_COLUMN);

        ArrayList selectedTiers = new ArrayList();

        // add selected tiers in the right order
        for (int i = 0; i < model.getRowCount(); i++) {
            Boolean include = (Boolean) model.getValueAt(i, includeCol);

            if (include.booleanValue()) {
                selectedTiers.add(((SelectableObject) model.getValueAt(i,
                        nameCol)).getValue());
            }
        }

        return selectedTiers;
    }

    /**
     * Shows an error dialog.
     *
     * @param message
     */
    private void showError(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Error"), JOptionPane.ERROR_MESSAGE);
    }

    /*
        private void specifyFieldSpecs() {
            ShoeboxMarkerDialog smd = new ShoeboxMarkerDialog(null, true);
            smd.setVisible(true);
            markers = smd.getMarkers();
        }
    */

    //***********************
    // inner classes
    //***********************	
    class VectorComparator implements Comparator {
        /**
         * Compares Vectors, on basis of their size. The largest one comes
         * first
         *
         * @see java.util.Comparator#compare(java.lang.Object,
         *      java.lang.Object)
         */
        public int compare(Object arg0, Object arg1) {
            Vector v0 = (Vector) arg0;
            Vector v1 = (Vector) arg1;

            if (v0.size() < v1.size()) {
                return 1;
            }

            if (v0.size() > v1.size()) {
                return -1;
            }

            return 0;
        }
    }

    /**
     * Renderer class that uses a different foreground color for selected objects.
     * @author Han Sloetjes
     */
    class MarkerCellRenderer extends DefaultTableCellRenderer {
        /**
         * Highlight the markers that should be followed by a whit line.
         * @see javax.swing.table.DefaultTableCellRenderer#getTableCellRendererComponent(javax.swing.JTable, java.lang.Object, boolean, boolean, int, int)
         */
        public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus, int row,
            int column) {
            Component c = super.getTableCellRendererComponent(table, value,
                    isSelected, hasFocus, row, column);

            if (value instanceof SelectableObject) {
                if (((SelectableObject) value).isSelected()) {
                    c.setForeground(Constants.ACTIVEANNOTATIONCOLOR);
                } else {
                    if (!isSelected) {
                        c.setForeground(table.getForeground());
                    }
                }
            }

            return c;
        }
    }
}
