/*
 * File:     ExportWordListDialog.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.util.ElanFileFilter;
import mpi.eudico.client.annotator.util.FileExtension;

import mpi.eudico.client.util.CheckBoxTableCellRenderer;
import mpi.eudico.client.util.Transcription2WordList;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.ParseException;
import mpi.eudico.server.corpora.clomimpl.abstr.TierImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.EAFSkeletonParser;

import java.awt.Frame;
import java.awt.GridBagConstraints;

import java.io.File;
import java.io.IOException;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import javax.swing.ButtonGroup;
import javax.swing.DefaultCellEditor;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JTextField;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;


/**
 * Tier selection dialog for export of a word list, either from a single file
 * or from multiple files.
 *
 * @author Han Sloetjes
 */
public class ExportWordListDialog extends AbstractTierExportDialog
    implements ChangeListener {
    /** Holds value of property DOCUMENT ME! */
    public static int WORDS = 0;

    /** Holds value of property DOCUMENT ME! */
    public static int ANNOTATIONS = 1;
    private List files;
    private JRadioButton customDelimRB;
    private JLabel tokenDelimLabel;
    private JRadioButton defaultDelimRB;
    private JTextField customDelimField;
    private ButtonGroup delimButtonGroup;
    private int mode = WORDS;

    /**
     * Constructor for single file.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param transcription the (single) transcription
     */
    public ExportWordListDialog(Frame parent, boolean modal,
        Transcription transcription) {
        super(parent, modal, transcription, null);
        makeLayout();
        extractTiers();
        postInit();
    }

    /**
     * Constructor for multiple files.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param files a list of eaf files
     */
    public ExportWordListDialog(Frame parent, boolean modal, List files) {
        this(parent, modal, files, WORDS);
    }

    /**
     * Constructor for multiple files.
     *
     * @param parent the parent frame
     * @param modal the modal flag
     * @param files a list of eaf files
     * @param mode the mode, WORDS or ANNOTATION export
     */
    public ExportWordListDialog(Frame parent, boolean modal, List files,
        int mode) {
        super(parent, modal, null, null);
        this.files = files;

        if ((mode == ANNOTATIONS) || (mode == WORDS)) {
            this.mode = mode;
        }

        makeLayout();
        extractTiersFromFiles();
        postInit();
    }

    /**
     * Extract candidate tiers for export.
     */
    protected void extractTiers() {
        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            if (transcription != null) {
                Vector v = transcription.getTiers();
                TierImpl t;

                for (int i = 0; i < v.size(); i++) {
                    t = (TierImpl) v.get(i);

                    model.addRow(new Object[] { Boolean.TRUE, t.getName() });
                }
            }
        }
    }

    /**
     * Extracts all unique tiers from multiple files.
     */
    private void extractTiersFromFiles() {
        if ((files == null) || (files.size() == 0)) {
            return;
        }

        if (model != null) {
            for (int i = model.getRowCount() - 1; i >= 0; i--) {
                model.removeRow(i);
            }

            ArrayList tierNames = new ArrayList();
            ArrayList pts = null;
            EAFSkeletonParser parser = null;
            String path;

            for (int i = 0; i < files.size(); i++) {
                path = ((File) files.get(i)).getAbsolutePath();

                try {
                    parser = new EAFSkeletonParser(path);
                    parser.parse();
                    pts = parser.getTiers();

                    TierImpl t;

                    for (int j = 0; j < pts.size(); j++) {
                        t = (TierImpl) pts.get(j);

                        if (!tierNames.contains(t.getName())) {
                            tierNames.add(t.getName());
                        }
                    }
                } catch (ParseException pe) {
                    LOG.warning(pe.getMessage());

                    //pe.printStackTrace();
                } catch (Exception ex) {
                    LOG.warning("Could not load file: " + path);
                }
            }

            for (int i = 0; i < tierNames.size(); i++) {
                model.addRow(new Object[] { Boolean.TRUE, tierNames.get(i) });
            }
        }
    }

    /**
     * Calls the super implementation and sets some properties of the tier
     * table.
     */
    protected void makeLayout() {
        super.makeLayout();

        model.setColumnIdentifiers(new String[] { EXPORT_COLUMN, TIER_NAME_COLUMN });
        tierTable.getColumn(EXPORT_COLUMN).setCellEditor(new DefaultCellEditor(
                new JCheckBox()));
        tierTable.getColumn(EXPORT_COLUMN).setCellRenderer(new CheckBoxTableCellRenderer());
        tierTable.getColumn(EXPORT_COLUMN).setMaxWidth(30);
        tierTable.setShowVerticalLines(false);
        tierTable.setTableHeader(null);

        // options
        if (mode == WORDS) {
            delimButtonGroup = new ButtonGroup();
            tokenDelimLabel = new JLabel();
            defaultDelimRB = new JRadioButton();
            customDelimRB = new JRadioButton();
            customDelimField = new JTextField();

            GridBagConstraints gridBagConstraints;
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 0;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.weightx = 1.0;
            gridBagConstraints.insets = insets;
            optionsPanel.add(tokenDelimLabel, gridBagConstraints);

            defaultDelimRB.setSelected(true);
            defaultDelimRB.addChangeListener(this);
            delimButtonGroup.add(defaultDelimRB);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 1;
            gridBagConstraints.gridwidth = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(defaultDelimRB, gridBagConstraints);

            customDelimRB.addChangeListener(this);
            delimButtonGroup.add(customDelimRB);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 0;
            gridBagConstraints.gridy = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(customDelimRB, gridBagConstraints);

            customDelimField.setEnabled(false);
            customDelimField.setColumns(6);
            gridBagConstraints = new GridBagConstraints();
            gridBagConstraints.gridx = 1;
            gridBagConstraints.gridy = 2;
            gridBagConstraints.anchor = GridBagConstraints.WEST;
            gridBagConstraints.insets = insets;
            optionsPanel.add(customDelimField, gridBagConstraints);
        } else {
            getContentPane().remove(optionsPanel);
        }

        updateLocale();
    }

    /**
     * Applies localized text values ui elements.
     */
    protected void updateLocale() {
        super.updateLocale();
        titleLabel.setText(ElanLocale.getString("ExportDialog.WordList.Title"));

        if (mode == WORDS) {
            titleLabel.setText(ElanLocale.getString(
                    "ExportDialog.WordList.Title"));
            tokenDelimLabel.setText(ElanLocale.getString(
                    "TokenizeDialog.Label.TokenDelimiter"));
            defaultDelimRB.setText(ElanLocale.getString("Button.Default") +
                "( . , ! ? \" \' )");
            customDelimRB.setText(ElanLocale.getString(
                    "TokenizeDialog.RadioButton.Custom"));
        } else if (mode == ANNOTATIONS) {
            titleLabel.setText(ElanLocale.getString(
                    "ExportDialog.AnnotationList.Title"));
        }
    }

    /**
     * @see mpi.eudico.client.annotator.export.AbstractBasicExportDialog#startExport()
     */
    protected boolean startExport() throws IOException {
        List selectedTiers = getSelectedTiers();

        if (selectedTiers.size() == 0) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportTradTranscript.Message.NoTiers"),
                ElanLocale.getString("Message.Warning"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        // prompt for file name and location
        File exportFile = null;

        if (mode == WORDS) {
            exportFile = promptForFile(ElanLocale.getString(
                        "ExportDialog.WordList.Title"), FileExtension.TEXT_EXT,
                    ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE),
                    true);
        } else if (mode == ANNOTATIONS) {
            exportFile = promptForFile(ElanLocale.getString(
                        "ExportDialog.AnnotationList.Title"),
                    FileExtension.TEXT_EXT,
                    ElanFileFilter.createFileFilter(ElanFileFilter.TEXT_TYPE),
                    true);
        }

        if (exportFile == null) {
            return false;
        }

        String delimiters = null;

        if ((mode == WORDS) && customDelimRB.isSelected()) {
            delimiters = customDelimField.getText();
        }

        Transcription2WordList twl = new Transcription2WordList();

        try {
            if (transcription != null) {
                twl.exportWords(transcription, selectedTiers, exportFile,
                    encoding, delimiters);
            } else {
                if (mode == ANNOTATIONS) {
                    twl.exportWords(files, selectedTiers, exportFile, encoding,
                        new String(""));
                } else if (mode == WORDS) {
                    twl.exportWords(files, selectedTiers, exportFile, encoding,
                        delimiters);
                }
            }
        } catch (IOException ioe) {
            JOptionPane.showMessageDialog(this,
                ElanLocale.getString("ExportDialog.Message.Error"),
                ElanLocale.getString("Message.Error"),
                JOptionPane.WARNING_MESSAGE);

            return false;
        }

        return true;
    }

    /**
     * The state changed event handling.
     *
     * @param ce the change event
     */
    public void stateChanged(ChangeEvent ce) {
        if (defaultDelimRB.isSelected()) {
            customDelimField.setEnabled(false);
        } else {
            customDelimField.setEnabled(true);
            customDelimField.requestFocus();
        }
    }
}
