/*
 * File:     ShortCutPrinter.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.export;

import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.util.TableSubHeaderObject;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.print.PageFormat;
import java.awt.print.Pageable;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;

import javax.swing.JOptionPane;
import javax.swing.JTable;


/**
 * Prints the contents of the shortcut table.
 */
public class ShortCutPrinter implements Printable, Pageable {
    private JTable table;
    private int numPages = 1;
    private int firstColWidth = 0;
    private PageFormat pf;
    private float scale = 0.9f;

    /**
     * Constructor.
     *
     * @param shortCutTable the shortcut table
     */
    public ShortCutPrinter(JTable shortCutTable) {
        table = shortCutTable;
    }

    /**
     * Initiates printing, shows page setup dialog and a print dialog.
     */
    public void startPrint() {
        PrinterJob printJob = PrinterJob.getPrinterJob();
        pf = printJob.pageDialog(printJob.defaultPage());
        calculateNumPages();
        printJob.setPrintable(this, pf);
        printJob.setPageable(this);

        if (printJob.printDialog()) {
            try {
                printJob.print();
            } catch (Exception ex) {
                ex.printStackTrace();
                JOptionPane.showMessageDialog(table,
                    ElanLocale.getString(
                        "InterlinearizerOptionsDlg.Error.Print") + " \n" + "(" +
                    ex.getMessage() + ")",
                    ElanLocale.getString("Message.Error"),
                    JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    /**
     * @see java.awt.print.Printable#print(java.awt.Graphics,
     *      java.awt.print.PageFormat, int)
     */
    public int print(Graphics graphics, PageFormat pageFormat, int pageIndex)
        throws PrinterException {
        if (pageIndex >= getNumberOfPages()) {
            return Printable.NO_SUCH_PAGE;
        }

        int[] rowExt = calculateRowExtremes(pageIndex);

        if (firstColWidth == 0) {
            firstColWidth = calcFirstColWidth(graphics);
        }

        renderPage(graphics, rowExt);

        return Printable.PAGE_EXISTS;
    }

    /**
     * @see java.awt.print.Pageable#getNumberOfPages()
     */
    public int getNumberOfPages() {
        return numPages;
    }

    /**
     * @see java.awt.print.Pageable#getPageFormat(int)
     */
    public PageFormat getPageFormat(int pageIndex)
        throws IndexOutOfBoundsException {
        if (pageIndex < 0) {
            throw new IndexOutOfBoundsException("Page Index: " + pageIndex +
                " < 0");
        }

        if (pageIndex >= getNumberOfPages()) {
            throw new IndexOutOfBoundsException("Page Index: " + pageIndex +
                " > " + (getNumberOfPages() - 1));
        }

        return pf;
    }

    /**
     * @see java.awt.print.Pageable#getPrintable(int)
     */
    public Printable getPrintable(int pageIndex)
        throws IndexOutOfBoundsException {
        if (pageIndex < 0) {
            throw new IndexOutOfBoundsException("Page Index: " + pageIndex +
                " < 0");
        }

        if (pageIndex >= getNumberOfPages()) {
            throw new IndexOutOfBoundsException("Page Index: " + pageIndex +
                " > " + (getNumberOfPages() - 1));
        }

        return this;
    }

    /**
     * Calculates the number of pages, based on the row heights. Line wrapping
     * is not yet  taken into account. Will probably not be  necessary.
     */
    private void calculateNumPages() {
        if ((table == null) || (pf == null)) {
            return;
        }

        int imgH = (int) pf.getImageableHeight();
        int curY = 0;

        curY += (int) (scale * table.getTableHeader().getHeight());

        for (int i = 0; i < table.getRowCount(); i++) {
            curY += (int) (scale * table.getRowHeight(i));

            if (curY > imgH) {
                numPages++;

                // start with this row on a new page
                i--;

                // if the last row is a sub header move it to the next page
                if (table.getValueAt(i, 0) instanceof TableSubHeaderObject) {
                    i--;
                }

                curY = 0;
            }
        }
    }

    /**
     * Calculates the number of pages, based on the row heights. Line wrapping
     * is not yet  taken into account. Will probably not be  necessary.
     *
     * @param page the index of the page
     *
     * @return an array of size 2, the first and last table row on this page
     */
    private int[] calculateRowExtremes(int page) {
        if ((table == null) || (pf == null)) {
            return null;
        }

        int[] extr = new int[] { 0, 0 };
        int imgH = (int) pf.getImageableHeight();
        int curY = 0;
        int curP = 0;
        curY += (int) (scale * table.getTableHeader().getHeight());

        for (int i = 0; i < table.getRowCount(); i++) {
            curY += (int) (scale * table.getRowHeight(i));

            if (curY > imgH) {
                curP++;

                // start with this row on a new page
                i--;

                // if the last row is a sub header move it to the next page
                if (table.getValueAt(i, 0) instanceof TableSubHeaderObject) {
                    i--;
                }

                if (curP == page) {
                    extr[0] = i;
                } else if (curP > page) {
                    extr[1] = i - 1;

                    return extr;
                }

                curY = 0;
            }

            if (i == (table.getRowCount() - 1)) {
                extr[1] = i;
            }
        }

        return extr;
    }

    /**
     * Calculates the width of the first column based on the length of the
     * strings in the shortcuts column
     *
     * @param graphics the printer graphics object
     *
     * @return the width for the first column
     */
    private int calcFirstColWidth(Graphics graphics) {
        if (table == null) {
            return 200;
        }

        int cw = 0;
        int rowW = 0;
        Object rv;
        Graphics2D g2d = (Graphics2D) graphics;

        // first calc the header text width
        String val = (String) table.getTableHeader().getColumnModel()
                                   .getColumn(0).getHeaderValue();

        cw = (int) (scale * g2d.getFontMetrics(table.getTableHeader().getFont())
                               .stringWidth(val));

        for (int i = 0; i < table.getRowCount(); i++) {
            rv = table.getValueAt(i, 0);

            // skip the sub-headers
            if (rv instanceof String) {
                val = (String) rv;
                rowW = (int) (scale * g2d.getFontMetrics(table.getFont())
                                         .stringWidth(val));

                if (rowW > cw) {
                    cw = rowW;
                }
            }
        }

        // add a margin
        return cw + 40;
    }

    /**
     * Renders the page, identified by the firts and last row to print, to the
     * graphics context.
     *
     * @param graphics the printer graphics
     * @param rowExt the row extremes, first and last row for this page
     */
    private void renderPage(Graphics graphics, int[] rowExt) {
        if ((table == null) || (pf == null)) {
            return;
        }

        Graphics2D g2d = (Graphics2D) graphics;
        g2d.setStroke(new BasicStroke(0.6f));

        int ix = (int) pf.getImageableX();
        int iy = (int) pf.getImageableY();
        int iw = (int) pf.getImageableWidth();
        int curY = 0;
        Object rv;
        String val;
        TableSubHeaderObject tsho;
        Font normF = table.getFont().deriveFont(Font.PLAIN,
                (int) (scale * table.getFont().getSize()));
        Font boldF = table.getFont().deriveFont(Font.BOLD,
                (int) (scale * (table.getFont().getSize() + 2)));

        if (rowExt[0] == 0) {
            // we have to print the headers, centered in the columns
            int tw = 0;
            val = (String) table.getTableHeader().getColumnModel().getColumn(0)
                                .getHeaderValue();
            tw = (int) (scale * g2d.getFontMetrics(table.getTableHeader()
                                                        .getFont()).stringWidth(val));

            int tx = ix + ((firstColWidth - tw) / 2);
            curY = iy + (int) (scale * table.getTableHeader().getHeight());
            g2d.setFont(normF);
            g2d.drawString(val, tx, curY - g2d.getFontMetrics().getDescent());
            val = (String) table.getTableHeader().getColumnModel().getColumn(1)
                                .getHeaderValue();
            tw = (int) (scale * g2d.getFontMetrics(table.getTableHeader()
                                                        .getFont()).stringWidth(val));
            tx = ix + firstColWidth + ((iw - firstColWidth - tw) / 2);
            g2d.drawString(val, tx, curY - g2d.getFontMetrics().getDescent());
            g2d.drawLine(ix, curY, ix + iw, curY);
        } else {
            curY += iy;
        }

        boolean gray = true;
        Color bg = new Color(236, 236, 236);
        int th = 0;

        for (int i = rowExt[0]; (i <= rowExt[1]) && (i < table.getRowCount());
                i++) {
            rv = table.getValueAt(i, 0);
            th = (int) (scale * table.getRowHeight(i));
            curY += th;

            if (rv instanceof String) {
                if (gray) {
                    g2d.setColor(bg);
                    g2d.fillRect(ix, curY - th, iw, th);
                }

                gray = !gray;
                g2d.setColor(Color.BLACK);
                g2d.drawLine(ix, curY - th, ix + iw, curY - th);
                g2d.setFont(normF);
                g2d.drawString((String) rv, ix + 4,
                    curY - g2d.getFontMetrics().getDescent());
                rv = table.getValueAt(i, 1);
                g2d.drawString((String) rv, ix + firstColWidth,
                    curY - g2d.getFontMetrics().getDescent());
            } else if (rv instanceof TableSubHeaderObject) {
                if (i != 0) {
                    g2d.drawLine(ix, curY - th, ix + iw, curY - th);
                }

                g2d.setFont(boldF);
                tsho = (TableSubHeaderObject) rv;
                g2d.drawString(tsho.toString(), ix + 4,
                    curY - g2d.getFontMetrics().getDescent() - 2);
                gray = true;
            }
        }
    }
}
