/*
 * File:     GridViewerPopupMenu.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.grid;

import mpi.eudico.client.annotator.Constants;
import mpi.eudico.client.annotator.ElanLocale;

import mpi.eudico.client.annotator.export.ExportGridTable;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.table.TableColumn;


/**
 * Popupmenu for AnnotationTable; for changing fontSize and Layout of the table
 * Extracted from GridViewer on Jun 30, 2004
 * @author Alexander Klassmann
 * @version Jun 30, 2004
 */
public class GridViewerPopupMenu extends JPopupMenu implements ActionListener {
    /** Holds value of property DOCUMENT ME! */
    final protected AnnotationTable table;

    /** Holds value of property DOCUMENT ME! */
    final private JMenu fontMenu;

    /** Holds value of property DOCUMENT ME! */
    final private JMenuItem toggleMenuItem;

    /** Holds value of property DOCUMENT ME! */
    final private JMenuItem exportMenuItem;

    /** Holds value of property DOCUMENT ME! */
    final private ButtonGroup fontSizeBG;
    private int fontSize;

    /** listeners that want to be notified of menu events */
    private List actionListeners;

    /**
     * Creates a new GridViewerPopupMenu instance
     *
     * @param table DOCUMENT ME!
     */
    public GridViewerPopupMenu(AnnotationTable table) {
        this.table = table;

        fontSizeBG = new ButtonGroup();
        fontMenu = new JMenu(ElanLocale.getString("Menu.View.FontSize"));

        JRadioButtonMenuItem fontRB;

        for (int i = 0; i < Constants.FONT_SIZES.length; i++) {
            fontRB = new JRadioButtonMenuItem(String.valueOf(
                        Constants.FONT_SIZES[i]));
            fontRB.setActionCommand("font" + Constants.FONT_SIZES[i]);

            if (table.getFont().getSize() == Constants.FONT_SIZES[i]) {
                fontRB.setSelected(true);
            }

            fontRB.addActionListener(this);
            fontSizeBG.add(fontRB);
            fontMenu.add(fontRB);
        }

        //add timeformat toggle
        toggleMenuItem = new JMenuItem(ElanLocale.getString(
                    "Menu.Options.TimeFormat"));
        toggleMenuItem.setActionCommand("TOGGLETIMEFORMAT");
        toggleMenuItem.addActionListener(this);

        exportMenuItem = new JMenuItem(ElanLocale.getString(
                    "Frame.GridFrame.ExportTableAsTab"));
        exportMenuItem.setActionCommand("EXPORT");
        exportMenuItem.addActionListener(this);
    }

    /**
     * The menu action event handling.
     *
     * @param e the action event
     */
    public void actionPerformed(ActionEvent e) {
        String actionCommand = e.getActionCommand();

        if (actionCommand.equals("TOGGLETIMEFORMAT")) {
            table.toggleTimeFormat();
        } else if (actionCommand.equals("EXPORT")) {
            ExportGridTable exporter = new ExportGridTable();
            exporter.exportTableAsTabDelimitedText(table);
        } else if (actionCommand.indexOf("font") != -1) {
            int index = actionCommand.indexOf("font") + 4;

            try {
                table.setFontSize(Integer.parseInt(actionCommand.substring(
                            index)));
                table.repaint();
            } catch (Exception ex) {
                //LOG.info("Could not set font size");
            }
        } else {
            table.setColumnVisible(actionCommand,
                ((JCheckBoxMenuItem) e.getSource()).isSelected());
            table.adjustAnnotationColumns();
        }

        table.doLayout();

        if (actionListeners != null) {
            for (int i = 0; i < actionListeners.size(); i++) {
                ((ActionListener) actionListeners.get(i)).actionPerformed(e);
            }
        }
    }

    /**
     * DOCUMENT ME!
     */
    protected void makeLayout() {
        removeAll();

        for (int i = 0; i < table.getColumnCount(); i++) {
            TableColumn column = table.getColumnModel().getColumn(i);
            String columnName = (String) column.getIdentifier();

            //Column ANNOTATION should not be removable and thus not in selection menu
            if (!columnName.equals(GridViewerTableModel.TIMEPOINT) &&
                    !columnName.equals(GridViewerTableModel.COUNT) &&
                    !columnName.equals(GridViewerTableModel.ANNOTATION)) {
                JMenuItem menuItem = new JCheckBoxMenuItem(column.getHeaderValue()
                                                                 .toString());
                menuItem.setActionCommand(columnName);
                menuItem.setSelected(table.isColumnVisible(columnName));
                menuItem.addActionListener(this);
                add(menuItem);
            }
        }

        addSeparator();
        add(fontMenu);
        addSeparator();
        add(toggleMenuItem);
        add(exportMenuItem);
    }

    /**
     * update Menu when it pops up
     * @see java.awt.Component#setVisible(boolean)
     */
    public void setVisible(boolean b) {
        if (b) {
            makeLayout();
        }

        super.setVisible(b);
    }

    /**
     * Returns the current font size.
     *
     * @return the current font size
     */
    public int getFontSize() {
        return fontSize;
    }

    /**
     * Sets the font size.
     *
     * @param size the new font size
     */
    public void setFontSize(int size) {
        fontSize = size;

        if (fontSizeBG != null) {
            Enumeration en = fontSizeBG.getElements();
            JMenuItem item;
            String value;

            while (en.hasMoreElements()) {
                item = (JMenuItem) en.nextElement();
                value = item.getText();

                try {
                    int v = Integer.parseInt(value);

                    if (v == fontSize) {
                        item.setSelected(true);

                        if (table != null) {
                            table.setFontSize(size);
                            table.repaint();
                        }

                        break;
                    }
                } catch (NumberFormatException nfe) {
                    //// do nothing
                }
            }
        }
    }

    /**
     * Apply localized strings to the menuitems.
     */
    public void updateLocale() {
        fontMenu.setText(ElanLocale.getString("Menu.View.FontSize"));
        toggleMenuItem.setText(ElanLocale.getString("Menu.Options.TimeFormat"));
        exportMenuItem.setText(ElanLocale.getString(
                "Frame.GridFrame.ExportTableAsTab"));
    }

    /**
     * Add action listener that will be notified of menu item actions <b>after</b>
     *  this class has dealt with the event.
     *
     * @param listener the listener
     */
    public void addActionListener(ActionListener listener) {
        if (actionListeners == null) {
            actionListeners = new ArrayList(2);
        }

        actionListeners.add(listener);
    }
}
