/*
 * File:     EditCVDialog.java
 * Project:  MPI Linguistic Application
 * Date:     12 December 2007
 *
 * Copyright (C) 2001-2008  Max Planck Institute for Psycholinguistics
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package mpi.eudico.client.annotator.gui;

import mpi.eudico.client.annotator.ElanLocale;
import mpi.eudico.client.annotator.Preferences;

import mpi.eudico.client.annotator.commands.Command;
import mpi.eudico.client.annotator.commands.ELANCommandFactory;

import mpi.eudico.client.annotator.util.ElanFileFilter;

import mpi.eudico.server.corpora.clom.Transcription;

import mpi.eudico.server.corpora.clomimpl.abstr.ParseException;
import mpi.eudico.server.corpora.clomimpl.abstr.TranscriptionImpl;
import mpi.eudico.server.corpora.clomimpl.dobes.EAFSkeletonParser;

import mpi.util.ControlledVocabulary;

import mpi.util.gui.AbstractEditCVDialog;

import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemListener;

import java.io.File;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.ComponentInputMap;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.border.TitledBorder;


/**
 * $Id: EditCVDialog.java 8258 2007-02-27 09:43:07Z hasloe $
 *
 * The Edit Controlled Vocabulary dialog is a dialog for defining and changing
 * controlled vocabularies and their entries.<br>
 *
 * @author Han Sloetjes, Alexander Klassmann
 * @version jun 04
 * @version Aug 2005 Identity removed
 * @version July 2006 refactored
 *
 */
public class EditCVDialog extends AbstractEditCVDialog implements ActionListener,
    ItemListener {
    /** a logger */
    private static final Logger LOG = Logger.getLogger(EditCVDialog.class.getName());
    private JButton importButton;
    private TranscriptionImpl transcription;

    /** Holds value of property DOCUMENT ME! */
    private final int SKIP = 0;

    /** Holds value of property DOCUMENT ME! */
    private final int REPLACE = 1;

    /** Holds value of property DOCUMENT ME! */
    private final int RENAME = 2;

    /** Holds value of property DOCUMENT ME! */
    private final int MERGE = 3;

    /**
     * Creates a new EditCVDialog.
     *
     * @param transcription the transcription containing the controlled
     *        vocabularies
     */
    public EditCVDialog(Transcription transcription) {
        super(ELANCommandFactory.getRootFrame(transcription), true, true,
            new ElanEditCVPanel());
        this.transcription = (TranscriptionImpl) transcription;
        addCloseActions();
        updateLabels();
        setPosition();
        updateComboBox();
        cvNameTextField.requestFocus();
    }

    /**
     * The button actions.
     *
     * @param actionEvent the actionEvent
     */
    public void actionPerformed(ActionEvent actionEvent) {
        Object source = actionEvent.getSource();

        // check source equality
        if (source == importButton) {
            importCV();
        } else {
            super.actionPerformed(actionEvent);
        }
    }

    /**
     * DOCUMENT ME!
     */
    public void updateLocale() {
        ((ElanEditCVPanel) cvEditorPanel).updateLabels();
        updateLabels();
    }

    /**
     * DOCUMENT ME!
     *
     * @return DOCUMENT ME!
     */
    protected List getCVList() {
        return transcription.getControlledVocabularies();
    }

    /**
     * Calls command to add a CV to the transcription. Re-initializes gui afterwords.
     */
    protected void addCV(String name) {
        //create a new CV and add it to the Transcription
        Command com = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.ADD_CV);
        Object[] args = new Object[2];
        args[0] = name;
        args[1] = cvDescArea.getText();

        com.execute(transcription, args);
        updateComboBox();
        cvComboBox.setSelectedIndex(cvComboBox.getItemCount() - 1);
    }

    /**
     * Calls command to change the CV in the transcription.
     */
    protected void changeCV(ControlledVocabulary cv, String name,
        String description) {
        // create a change CV command
        Command com = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.CHANGE_CV);
        Object[] args = new Object[4];
        args[0] = oldCVName;
        args[1] = (description != null) ? description : oldCVDesc;
        args[2] = (name != null) ? name : oldCVName;
        args[3] = description;

        com.execute(transcription, args);

        updateComboBox();
        cvComboBox.setSelectedItem(cv);
    }

    /**
     * If there are any tiers using this
     * cv, a message will ask the user for confirmation.
     * (Overrides condition of super class)
     */
    protected void deleteCV() {
        ControlledVocabulary conVoc = (ControlledVocabulary) cvComboBox.getSelectedItem();

        // warn if there are tiers using lin. types using this cv
        if (transcription.getTiersWithCV(conVoc.getName()).size() > 0) {
            String mes = ElanLocale.getString("EditCVDialog.Message.CVInUse") +
                "\n" +
                ElanLocale.getString("EditCVDialog.Message.CVConfirmDelete");

            if (!showConfirmDialog(mes)) {
                return;
            }
        }

        deleteCV(conVoc);
    }

    /**
     * Calls command to delete CV in the transcription. Re-initializes gui afterwords.
     */
    protected void deleteCV(ControlledVocabulary cv) {
        Command com = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.DELETE_CV);
        com.execute(transcription, new Object[] { cv });
        updateComboBox();

        if (cvComboBox.getItemCount() == 0) {
            cvEditorPanel.setControlledVocabulary(null);
        }
    }

    /**
     * This method is called from within the constructor to initialize the
     * dialog's components.
     */
    protected void makeLayout() {
        super.makeLayout();
        importButton = new JButton();
        importButton.addActionListener(this);
        cvButtonPanel.add(importButton);
    }

    /**
     * Shows a confirm (yes/no) dialog with the specified message string.
     *
     * @param message the messsage to display
     *
     * @return true if the user clicked OK, false otherwise
     */
    protected boolean showConfirmDialog(String message) {
        int confirm = JOptionPane.showConfirmDialog(this, message,
                ElanLocale.getString("Message.Warning"),
                JOptionPane.YES_NO_OPTION);

        return confirm == JOptionPane.YES_OPTION;
    }

    /**
     * Shows a warning/error dialog with the specified message string.
     *
     * @param message the message to display
     */
    protected void showWarningDialog(String message) {
        JOptionPane.showMessageDialog(this, message,
            ElanLocale.getString("Message.Warning"), JOptionPane.WARNING_MESSAGE);
    }

    /**
     * Since this dialog is meant to be moda l a Locale change while this dialog
     * is open  is not supposed to happen. This will set the labels etc. using
     * the current locale  strings.
     */
    protected void updateLabels() {
        closeDialogButton.setText(ElanLocale.getString(
                "EditCVDialog.Button.Close"));
        deleteCVButton.setText(ElanLocale.getString("Button.Delete"));
        changeCVButton.setText(ElanLocale.getString("Button.Change"));
        addCVButton.setText(ElanLocale.getString("Button.Add"));
        importButton.setText(ElanLocale.getString("Button.Import"));
        cvDescLabel.setText(ElanLocale.getString(
                "EditCVDialog.Label.CVDescription"));
        cvNameLabel.setText(ElanLocale.getString("EditCVDialog.Label.Name"));
        cvPanel.setBorder(new TitledBorder(ElanLocale.getString(
                    "EditCVDialog.Label.CV")));
        currentCVLabel.setText(ElanLocale.getString(
                "EditCVDialog.Label.Current"));
        titleLabel.setText(ElanLocale.getString("EditCVDialog.Title"));
        setTitle(ElanLocale.getString("EditCVDialog.Title"));

        cvNameExistsMessage = ElanLocale.getString(
                "EditCVDialog.Message.CVExists");
        cvInvalidNameMessage = ElanLocale.getString(
                "EditCVDialog.Message.CVValidName");
        cvContainsEntriesMessage = ElanLocale.getString(
                "EditCVDialog.Message.CVInUse");
        deleteQuestion = ElanLocale.getString(
                "EditCVDialog.Message.CVConfirmDelete");
    }

    /**
     * Prompts the user to select an template import file.
     *
     * @return the template file, or null when no valid file was selected
     */
    private File getImportFile() {
        // setup a file chooser
        String dir = (String) Preferences.get("LastUsedEAFDir", null);

        if (dir == null) {
            dir = (new File(transcription.getName())).getParent();

            if (dir == null) {
                dir = System.getProperty("user.dir");
            }
        }

        JFileChooser chooser = new JFileChooser(dir);
        chooser.setDialogTitle(ElanLocale.getString("Button.Import"));
        chooser.removeChoosableFileFilter(chooser.getAcceptAllFileFilter());
        chooser.addChoosableFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.EAF_TYPE));
        chooser.setFileFilter(ElanFileFilter.createFileFilter(
                ElanFileFilter.TEMPLATE_TYPE));

        JTextArea textAr = new JTextArea(ElanLocale.getString(
                    "EditCVDialog.Message.Browse"));
        textAr.setBackground(chooser.getBackground());
        textAr.setWrapStyleWord(true);
        textAr.setLineWrap(true);
        textAr.setEditable(false);
        textAr.setPreferredSize(new Dimension(160, 100));
        textAr.setMargin(new Insets(5, 5, 5, 5));

        chooser.setAccessory(textAr);

        int option = chooser.showOpenDialog(this);

        if (option == JFileChooser.CANCEL_OPTION) {
            return null;
        } else if (option == JFileChooser.APPROVE_OPTION) {
            File impFile = chooser.getSelectedFile();
            String filePath = impFile.getAbsolutePath();

            if (!impFile.exists() || impFile.isDirectory()) {
                StringBuffer strMessage = new StringBuffer(ElanLocale.getString(
                            "Menu.Dialog.Message1"));
                strMessage.append(impFile.getName());
                strMessage.append(ElanLocale.getString("Menu.Dialog.Message2"));

                String strError = ElanLocale.getString("Message.Error");
                JOptionPane.showMessageDialog(this, strMessage, strError,
                    JOptionPane.ERROR_MESSAGE);

                return null;
            }

            if (!filePath.toLowerCase().endsWith(".etf") &&
                    !filePath.toLowerCase().endsWith(".eaf")) {
                StringBuffer strMessage = new StringBuffer(ElanLocale.getString(
                            "Menu.Dialog.Message1"));
                strMessage.append(impFile.getName());
                strMessage.append(ElanLocale.getString("Menu.Dialog.Message3"));

                String strError = ElanLocale.getString("Message.Error");
                JOptionPane.showMessageDialog(this, strMessage, strError,
                    JOptionPane.ERROR_MESSAGE);

                return null;
            }

            Preferences.set("LastUsedEAFDir", impFile.getParent(), null);

            return impFile;
        }

        return null;
    }

    /**
     * Adds an existing, imported CV, probably already containing entries to
     * the  Transcription.
     *
     * @param conVoc the CV to add
     *
     * @see #importCV()
     */
    private void addCV(ControlledVocabulary conVoc) {
        if (conVoc == null) {
            return;
        }

        //create a new CV and add it to the Transcription
        Command com = ELANCommandFactory.createCommand(transcription,
                ELANCommandFactory.ADD_CV);
        Object[] args = new Object[1];
        args[0] = conVoc;

        com.execute(transcription, args);

        //reextractCVs() is done somewhere else
    }

    /**
     * Imports a or several CV's from a template file, selected by the user.
     * When there are CV's with the same identifier as existing CV's the user
     * is prompted to either replace an existing CV or rename the imported CV
     * or just skip the CV.
     */
    private void importCV() {
        File importFile = getImportFile();

        if (importFile == null) {
            return;
        }

        // use an eaf skeleton parser, can handle etf as well as eaf
        EAFSkeletonParser skelParser = null;

        try {
            skelParser = new EAFSkeletonParser(importFile.getAbsolutePath());
            skelParser.parse();
        } catch (ParseException pe) {
            showWarningDialog(ElanLocale.getString(
                    "EditCVDialog.Message.ReadError") +
                importFile.getAbsolutePath());

            LOG.warning("Could not parse the file: " + pe.getMessage());

            return;
        }

        ArrayList allCVs = skelParser.getControlledVocabularies();

        if (allCVs.size() == 0) {
            showWarningDialog(ElanLocale.getString(
                    "EditCVDialog.Message.NoCVFound"));

            return;
        }

        ControlledVocabulary cv = null;

        //now add them to the transcription, ensuring that all CV-names are unique
        for (int i = 0; i < allCVs.size(); i++) {
            cv = (ControlledVocabulary) allCVs.get(i);

            if (transcription.getControlledVocabulary(cv.getName()) != null) {
                // cv with that name already exists: prompt user:
                // replace, rename, merge or skip
                int option = showCVQuestionDialog(cv.getName());

                if (option == REPLACE) {
                    replaceCV(cv);
                } else if (option == RENAME) {
                    String newName;

                    while (true) {
                        newName = showAskNameDialog(cv.getName());

                        if (transcription.getControlledVocabulary(newName) != null) {
                            showWarningDialog(ElanLocale.getString(
                                    "EditCVDialog.Message.CVExists"));

                            continue;
                        }

                        break;
                    }

                    if ((newName == null) || (newName.length() == 0)) {
                        continue; //means skipping
                    }

                    cv.setName(newName);
                    addCV(cv);
                } else if (option == MERGE) {
                    mergeCVs(cv);
                }

                // else continue...
            } else {
                // the transcription does not contain a cv with the same name, add it
                addCV(cv);
            }
        }

        updateComboBox();
    }

    /**
     * Deletes an existing CV with the name of the specified CV and adds  the
     * specified CV. The user is not promted or warned.
     *
     * @param conVoc the new ControlledVocabulary
     */
    private void replaceCV(ControlledVocabulary conVoc) {
        String name = conVoc.getName();
        ControlledVocabulary oldCv = transcription.getControlledVocabulary(name);

        if (oldCv != null) {
            Command com = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.REPLACE_CV);
            com.execute(transcription, new Object[] { oldCv, conVoc });
        }
    }

    /**
     * Merges two CV's with the same name; entries present in the second cv that are not in the first cv
     * are added to the first cv.
     *
     * @param conVoc the second cv
     */
    private void mergeCVs(ControlledVocabulary conVoc) {
        String name = conVoc.getName();
        ControlledVocabulary oldCv = transcription.getControlledVocabulary(name);

        if (oldCv != null) {
            Command com = ELANCommandFactory.createCommand(transcription,
                    ELANCommandFactory.MERGE_CVS);
            com.execute(transcription, new Object[] { oldCv, conVoc });
        }
    }

    /**
     * Prompts the user to enter a new name for a CV.
     *
     * @param name the old name
     *
     * @return the new name, or null when the user has cancelled the dialog
     */
    private String showAskNameDialog(String name) {
        String message = ElanLocale.getString("EditCVDialog.Message.NewName") +
            "\n\n- " + name;
        String newName = JOptionPane.showInputDialog(this, message,
                ElanLocale.getString("EditCVDialog.Message.Rename"),
                JOptionPane.QUESTION_MESSAGE);

        return newName;
    }

    /**
     * Ask the user to skip, replace existing cv or rename importing cv.
     *
     * @param cvName name of the cv
     *
     * @return 0 means skip, 1 means replace and 2 means rename
     */
    private int showCVQuestionDialog(String cvName) {
        String[] options = new String[4];
        options[0] = ElanLocale.getString("EditCVDialog.Message.Skip");
        options[1] = ElanLocale.getString("EditCVDialog.Message.Replace");
        options[2] = ElanLocale.getString("EditCVDialog.Message.Rename");
        options[3] = ElanLocale.getString("EditCVDialog.Message.Merge");

        String message = ElanLocale.getString("EditCVDialog.Message.CVExists") +
            "\n\n- " + cvName + "\n";

        JOptionPane pane = new JOptionPane(message,
                JOptionPane.QUESTION_MESSAGE, JOptionPane.DEFAULT_OPTION, null,
                options);
        pane.createDialog(this, "").show();

        Object selValue = pane.getValue();

        for (int i = 0; i < options.length; i++) {
            if (selValue == options[i]) {
                return i;
            }
        }

        return SKIP;
    }

    /**
     * Add the Escape and Ctrl-W close actions.
     */
    protected void addCloseActions() {
        EscCloseAction escAction = new EscCloseAction(this);
        CtrlWCloseAction wAction = new CtrlWCloseAction(this);

        InputMap inputMap = getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW);
        ActionMap actionMap = getRootPane().getActionMap();

        if (inputMap instanceof ComponentInputMap && (actionMap != null)) {
            String esc = "esc";
            inputMap.put((KeyStroke) escAction.getValue(Action.ACCELERATOR_KEY),
                esc);
            actionMap.put(esc, escAction);

            String wcl = "cw";
            inputMap.put((KeyStroke) wAction.getValue(Action.ACCELERATOR_KEY),
                wcl);
            actionMap.put(wcl, wAction);
        }
    }
}
